﻿using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using System;
using System.Collections.Generic;


namespace GameResources.Logic.Physics
{
    public class PhysicsEngine
    {
        private readonly IMoveableObject moveableObject;
        private readonly WorldMapCell worldMapCell;
        private readonly GameState gameState;
        public bool IgnoreAllCollisions { get; set; }
        public bool StraightOnly { get; set; }
        public IList<IMoveableObject> IgnoreObjects { get; private set; }


        public PhysicsEngine(IMoveableObject moveableObject, WorldMapCell worldMapCell, GameState gameState, bool straightOnly)
        {
            this.moveableObject = moveableObject;
            this.worldMapCell = worldMapCell;
            this.gameState = gameState;
            IgnoreAllCollisions = false;
            IgnoreObjects = new List<IMoveableObject>();
            StraightOnly = straightOnly;
        }


        public void Update()
        {
            IList<(long, long)> possibleMoves = GetPossibleMoves();
            long originalX = moveableObject.Position.PositionX;
            long originalY = moveableObject.Position.PositionY;

            foreach ((long, long) moves in possibleMoves)
            {
                if (CanMoveToLocation(moves.Item1, moves.Item2))
                {
                    MoveObject(moves.Item1, moves.Item2);

                    break;
                }
            }

            double newX = moveableObject.Position.PositionX;
            double newY = moveableObject.Position.PositionY;

            AdjustSpeed(originalX, originalY, newX, newY);

            moveableObject.Motion.IsOnGround = IsOnGround();
        }


        private void AdjustSpeed(double oldX, double oldY, double newX, double newY)
        {
            Motion motion = moveableObject.Motion;

            //cases - stop
            if (StraightOnly)
            {
                if (newX == oldX && motion.VelocityX != 0 || newY == oldY && motion.VelocityY != 0)
                {
                    motion.VelocityX = 0;
                    motion.VelocityY = 0;
                }
            }
            else
            {
                if (newX == oldX && motion.VelocityX != 0) motion.VelocityX = 0;
                if (newY == oldY && motion.VelocityY != 0) motion.VelocityY = 0;
            }
        }


        private bool IsOnGround()
        {
            if ((int)moveableObject.Position.PositionY == worldMapCell.GetMaxTileY() * TileSet.DRAW_SIZE) return true;

            long x = moveableObject.Position.PositionX;
            long y = moveableObject.Position.PositionY + 1;

            CollisionDetection collisionDetection = new CollisionDetection(moveableObject, worldMapCell, gameState);
            return collisionDetection.HasCollidedInWorldAtPosition(x, y);
        }


        private IList<(long, long)> GetPossibleMoves()
        {
            IList<(long, long)> points = new List<(long, long)>();
            Motion motion = moveableObject.Motion;

            int mx = (int)Math.Round(Math.Abs(motion.VelocityX));
            int my = (int)Math.Round(Math.Abs(motion.VelocityY));
            int sx = motion.VelocityX < 0 ? -1 : 1;
            int sy = motion.VelocityY < 0 ? -1 : 1;

            for (long x = mx; x >= 0; x -= 1)
            {
                for (long y = my; y >= 0; y -= 1)
                {
                    if (StraightOnly && x == y || StraightOnly && x == 0 || StraightOnly && y == 0 || StraightOnly == false)
                    {
                        (long, long) point = (x * sx, y * sy);

                        points.Add(point);
                    }
                }
            }

            return points;
        }



        private bool CanMoveToLocation(long dx, long dy)
        {
            long possibleNewPositionX = moveableObject.Position.PositionX + dx;
            long possibleNewPositionY = moveableObject.Position.PositionY + dy;

            Rectangle objectBounds = moveableObject.GetCollisionAreaAtPosition(possibleNewPositionX, possibleNewPositionY);

            //tiles
            if (IgnoreAllCollisions == false)
            {
                CollisionDetection collisionDetection = new CollisionDetection(moveableObject, worldMapCell, gameState);
                if (collisionDetection.HasCollidedInWorldAtPosition(possibleNewPositionX, possibleNewPositionY))
                {
                    return false;
                }
            }

            //npc's
            if (IgnoreAllCollisions == false)
            {
                //player
                if (moveableObject != gameState.Player && IgnoreObjects.Contains(gameState.Player) == false)
                {
                    if (objectBounds.Intersects(gameState.Player.GetCurrentCollisionArea()))
                    {
                        return false;
                    }
                }
            }

            return true;
        }


        private void MoveObject(long dx, long dy)
        {
            moveableObject.Position.PositionX += dx;
            moveableObject.Position.PositionY += dy;
        }


    }
}
