﻿using GameResources.Manager;
using GameResources.Models.Characters;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Repository;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameResources.Models.Weapons
{
    public class Bomb : IWeaponSprite
    {
        public WorldPosition Position { get; set; }
        public Motion Motion { get; set; }
        public bool IsExpired { get; set; }
        public DamageType DamageType { get; private set; }
        public WeaponModifier WeaponModifier { get; set; }
        public int Damage { get; private set; }

        private readonly int size;
        private readonly IList<Texture2D> textures;
        private int activeTexture;
        private readonly IBattleObject caster;

        private long lastAnimationUpdate;
        private long creationTime;


        public Bomb(IGameNamedContentRepository<Texture2D> textureRepository, IBattleObject caster)
        {
            this.textures = new List<Texture2D>();
            this.textures.Add(textureRepository.GetContentByName(@"textures\sprites\bomb-1"));
            this.textures.Add(textureRepository.GetContentByName(@"textures\sprites\bomb-2"));
            this.textures.Add(textureRepository.GetContentByName(@"textures\sprites\bomb-3"));
            this.textures.Add(textureRepository.GetContentByName(@"textures\sprites\bomb-2"));
            this.activeTexture = 0;

            this.size = 16;
            this.caster = caster;
            this.Damage = 0;

            this.IsExpired = false;
            this.Motion = new Motion();
            this.Position = new WorldPosition();
            this.DamageType = DamageType.Missile;
            this.WeaponModifier = new WeaponModifier();

            this.lastAnimationUpdate = 0;
            this.creationTime = 0;
        }


        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            Texture2D texture = this.textures[activeTexture];

            Rectangle rect = new Rectangle((int)(Position.PositionX - dx - (size / 2)), (int)(Position.PositionY - dy - (size / 2)), size, size);
            spriteBatch.Draw(texture, rect, Color.White);
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            creationTime += ticks;
            lastAnimationUpdate += ticks;

            //animate object
            if (lastAnimationUpdate >= (TimeSpan.TicksPerSecond * 0.1))
            {
                UpdateAnimation();

                lastAnimationUpdate = 0;
            }

            //expire sprite
            if (creationTime >= (TimeSpan.TicksPerSecond * 1.0))
            {
                //Bomb Explosion weapon sprite (damage causing)
                BombExplosion explosion = new BombExplosion(gameState, gameContentManager, this.caster);
                explosion.Position = this.Position.Copy();
                gameState.ActiveWeaponSprites.Add(explosion);

                gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Bomb Explosion").Play();

                IsExpired = true;
            }
        }


        private void UpdateAnimation()
        {
            activeTexture++;

            if (activeTexture == this.textures.Count) activeTexture = 0;
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(Position.PositionX, Position.PositionY);
        }



        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            Rectangle area = new Rectangle((int)(x - 1), (int)(y - 1), 3, 3);

            return area;
        }


    }
}
