﻿using GameResources.Manager;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Repository;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameResources.Models.World
{
    public class Item : IMoveableObject
    {
        public WorldPosition Position { get; set; }
        public Motion Motion { get; set; }
        public ItemPlacementDefinition ItemPlacementDefinition { get; private set; }
        public ItemType ItemType { get; private set; }

        private readonly IGameContentManager gameContentManager;
        private readonly IList<Texture2D> textures;
        
        private int activeTexture;
        private long lastAnimationUpdate;
        private int size;

        public Item(ItemPlacementDefinition definition, IGameContentManager gameContentManager)
        {
            this.ItemPlacementDefinition = definition;
            this.gameContentManager = gameContentManager;
            this.ItemType = definition.ItemType;
            this.Motion = new Motion();

            this.Position = new WorldPosition()
            {
                WorldMapKey = definition.WorldKey,
                WorldCellX = definition.CellX,
                WorldCellY = definition.CellY,
                PositionX = definition.X * TileSet.DRAW_SIZE,
                PositionY = definition.Y * TileSet.DRAW_SIZE
            };

            this.activeTexture = 0;
            this.textures = new List<Texture2D>();

            this.lastAnimationUpdate = 0;

            LoadTextures();
        }


        public Item(ItemType itemType, IGameContentManager gameContentManager, long positionX, long positionY)
        {
            this.ItemPlacementDefinition = null;
            this.gameContentManager = gameContentManager;
            this.ItemType = itemType;
            this.Position = new WorldPosition()
            {
                PositionX = positionX,
                PositionY = positionY
            };
            this.Motion = new Motion();

            this.activeTexture = 0;
            this.textures = new List<Texture2D>();

            this.lastAnimationUpdate = 0;

            LoadTextures();
        }


        private void LoadTextures()
        {
            IGameNamedContentRepository<Texture2D> textureRepository = this.gameContentManager.TextureRepository;

            this.size = 32;

            if (this.ItemType == ItemType.Bombs)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\bomb-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\bomb-2"));
            }
            else if (this.ItemType == ItemType.ChargeBeam)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\chargebeam-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\chargebeam-2"));
            }
            else if (this.ItemType == ItemType.EnergyTank)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\energytank-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\energytank-2"));
            }
            else if (this.ItemType == ItemType.PickupHealthLarge)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-large-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-large-2"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-large-3"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-large-4"));
            }
            else if (this.ItemType == ItemType.PickupHealthSmall)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-small-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-small-2"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-small-3"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\health-small-4"));

                this.size = 16;
            }
            else if (this.ItemType == ItemType.IceBeam)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\icebeam-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\icebeam-2"));
            }
            else if (this.ItemType == ItemType.Missiles)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\missile-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\missile-2"));
            }
            else if (this.ItemType == ItemType.MorphBall)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\morphball-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\morphball-2"));
            }
            else if (this.ItemType == ItemType.PlasmaBeam)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\plasmabeam-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\plasmabeam-2"));
            }
            else if (this.ItemType == ItemType.SpazerBeam)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\spazerbeam-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\spazerbeam-2"));
            }
            else if (this.ItemType == ItemType.SpringBall)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\springball-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\springball-2"));
            }
            else if (this.ItemType == ItemType.SuperBombs)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\superbomb-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\superbomb-2"));
            }
            else if (this.ItemType == ItemType.SuperJump)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\superjump-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\superjump-2"));
            }
            else if (this.ItemType == ItemType.SuperMissiles)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\supermissile-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\supermissile-2"));
            }
            else if (this.ItemType == ItemType.VariaSuit)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\varia-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\varia-2"));
            }
            else if (this.ItemType == ItemType.WaveBeam)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\wavebeam-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\wavebeam-2"));
            }
            else if (this.ItemType == ItemType.GravitySuit)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\gravity-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\gravity-2"));
            }
            else if (this.ItemType == ItemType.SpaceJump)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\spacejump-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\spacejump-2"));
            }
            else if (this.ItemType == ItemType.ScrewAttack)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\screwattack-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\screwattack-2"));
            }
            else if (this.ItemType == ItemType.SpeedBoost)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\speedboost-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\speedboost-2"));
            }
            else if (this.ItemType == ItemType.PickupMissiles)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\pickup-missile-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\pickup-missile-2"));
            }
            else if (this.ItemType == ItemType.PickupSuperMissiles)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\pickup-supermissile-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\pickup-supermissile-2"));
            }
            else if (this.ItemType == ItemType.PickupSuperBombs)
            {
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\pickup-superbomb-1"));
                this.textures.Add(textureRepository.GetContentByName(@"textures\items\pickup-superbomb-2"));
            }
        }


        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            

            Rectangle rect = new Rectangle((int)(this.Position.PositionX - dx), (int)(this.Position.PositionY - dy), size, size);



            spriteBatch.Draw(this.textures[activeTexture], rect, Color.White);
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            lastAnimationUpdate += ticks;

            if(lastAnimationUpdate >= (TimeSpan.TicksPerSecond * 0.3))
            {
                activeTexture++;
                if (activeTexture >= this.textures.Count) activeTexture = 0;
                lastAnimationUpdate = 0;
            }

        }


        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            return new Rectangle((int)x, (int)y, TileSet.DRAW_SIZE, TileSet.DRAW_SIZE);
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(Position.PositionX, Position.PositionY);
        }


        public Texture2D GetDisplayTexture()
        {
            return this.textures[0];
        }


        public string GetDisplayMessage()
        {
            if (this.ItemType == ItemType.Bombs) return "Found bombs!\nShoot when in morph ball";
            else if (this.ItemType == ItemType.ChargeBeam) return "Found charge beam!\nHold shoot, wait for charge and release";
            else if (this.ItemType == ItemType.EnergyTank) return "Found an energy tank!\nHealth increased by 100";
            else if (this.ItemType == ItemType.IceBeam) return "Found ice beam!\nCan freeze enemies";
            else if (this.ItemType == ItemType.Missiles) return "Found missiles!\nMissile capacity increased by " + GetQuantity();
            else if (this.ItemType == ItemType.MorphBall) return "Found morph ball!\nPress down twice";
            else if (this.ItemType == ItemType.PlasmaBeam) return "Found plasma beam!\nCan shoot through enemies";
            else if (this.ItemType == ItemType.SpazerBeam) return "Found spazer beam!\nFires 3 shots at once";
            else if (this.ItemType == ItemType.SpringBall) return "Found spring ball!\nJump when in morph ball";
            else if (this.ItemType == ItemType.SuperBombs) return "Found super bombs!\nSuper bomb capacity increased by " + GetQuantity();
            else if (this.ItemType == ItemType.SuperJump) return "Found super jump!\nCan jump higher";
            else if (this.ItemType == ItemType.SuperMissiles) return "Found super missiles!\nSuper missile capacity increased by " + GetQuantity();
            else if (this.ItemType == ItemType.VariaSuit) return "Found varia suit!\nCan resist heat";
            else if (this.ItemType == ItemType.WaveBeam) return "Found wave beam!\nCan shoot through walls";
            else if (this.ItemType == ItemType.GravitySuit) return "Found gravity suit!\nCan now move normally in water";
            else if (this.ItemType == ItemType.SpaceJump) return "Found space jump!\nCan now jump unlimited times";
            else if (this.ItemType == ItemType.ScrewAttack) return "Found screw attack!\nJumping attacks enemies";
            else if (this.ItemType == ItemType.SpeedBoost) return "Found speed boost!\nStart running to get super speed";
            else return "";
        }

        
        public int GetQuantity()
        {
            if (this.ItemType == ItemType.Missiles) return 5;
            else if (this.ItemType == ItemType.SuperMissiles) return 2;
            else if (this.ItemType == ItemType.SuperBombs) return 2;
            else if (this.ItemType == ItemType.EnergyTank) return 100;
            else if (this.ItemType == ItemType.PickupHealthLarge) return 33;
            else if (this.ItemType == ItemType.PickupHealthSmall) return 11;
            else if (this.ItemType == ItemType.PickupMissiles) return 2;
            else if (this.ItemType == ItemType.PickupSuperMissiles) return 2;
            else if (this.ItemType == ItemType.PickupSuperBombs) return 2;
            else return 0;
        }


    }
}
