﻿using Microsoft.Xna.Framework.Content;
using RawResources.Models;
using RawResources.Repository;
using System.Collections.Generic;


namespace GameResources.Repository
{
    public class CachedRawResourceRepository<T> : IGameNamedContentRepository<T>
    {
        private readonly IRawContentRepository<RawResource> rawContentRepository;
        private readonly ContentManager contentManager;
        private readonly IDictionary<int, T> cacheByKey;
        private readonly IDictionary<string, T> cacheByName;


        public CachedRawResourceRepository(ContentManager contentManager, IRawContentRepository<RawResource> rawContentRepository)
        {
            this.rawContentRepository = rawContentRepository;
            this.contentManager = contentManager;
            this.cacheByKey = new Dictionary<int, T>();
            this.cacheByName = new Dictionary<string, T>();
        }


        public T GetContentByKey(int key)
        {
            if (this.cacheByKey.ContainsKey(key)) return this.cacheByKey[key];

            RawResource rawResource = this.rawContentRepository.GetContent(key);
            if (rawResource == null) return default;

            Load(rawResource);

            return this.cacheByKey[key];
        }


        public T GetContentByName(string name)
        {
            if (this.cacheByName.ContainsKey(name)) return this.cacheByName[name];

            RawResource rawResource = null;

            IList<int> keys = this.rawContentRepository.GetKeys();
            for(int i = 0; i < keys.Count; i++)
            {
                int key = keys[i];

                RawResource r = this.rawContentRepository.GetContent(key);
                
                if(r.FilePath.Equals(name))
                {
                    rawResource = r;
                }
            }

            if (rawResource == null) return default;

            Load(rawResource);

            return this.cacheByName[name];
        }


        private void Load(RawResource rawResource)
        {
            T item = this.contentManager.Load<T>(rawResource.FilePath);
            this.cacheByKey.Add(rawResource.Key, item);
            this.cacheByName.Add(rawResource.FilePath, item);
        }


    }
}
