﻿using GameResources.Manager;
using GameResources.Models.Game;
using GameResources.Models.Samus;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Media;
using System.Collections.Generic;
using UserInterface.Core;
using UserInterface.Helper;
using UserInterface.Screen;


namespace UserInterface.Control
{
    public class GameViewer : IControl
    {
        public IList<IControl> Controls { get; set; }
        public Rectangle Location { get; set; }
        public bool IsSelected { get; set; }
        public bool IsSelectable { get; }

        private readonly IGameContentManager gameContentManager;
        private readonly IScreenController screenController;
        private GameState gameState;
        private PlayerHud playerHud;
        private MiniMap miniMap;
        private Camera camera;
        private readonly Song bossSong; 

        public GameViewer(IGameContentManager gameContentManager, IScreenController screenController)
        {
            this.Controls = new List<IControl>();
            IsSelectable = false;
            IsSelected = false;

            this.screenController = screenController;
            this.gameContentManager = gameContentManager;

            bossSong = gameContentManager.SongRepository.GetContentByName(@"music\sm-boss");

            Init();
        }


        private void Init()
        {
            this.camera = new Camera();

            playerHud = new PlayerHud(this.gameContentManager.TextureRepository, this.gameContentManager.FontRepository)
            {
                Location = new Rectangle(10, 10, 1, 1)
            };
            Controls.Add(playerHud);

            miniMap = new MiniMap(this.gameContentManager.TextureRepository, this.gameContentManager.FontRepository, 5, Color.Black)
            {
                Location = new Rectangle(Resolution.RESOLUTION_X - 80 - 10, 10, 80, 80)
            };
            Controls.Add(miniMap);
        }


        public GameState GameState
        {
            get
            {
                return gameState;
            }
            set
            {
                this.gameState = value;

                if(this.gameState != null) this.playerHud.Player = this.gameState.Player;
                if (this.gameState != null) this.miniMap.GameState = this.gameState;
            }
        }


        public void Draw(SpriteBatch spriteBatch)
        {
            if (gameState == null) return;

            DrawScreen(spriteBatch);

            for(int i = 0; i < this.Controls.Count; i++) 
            {
                IControl control = this.Controls[i];
                control.Draw(spriteBatch);
            }
        }


        private void DrawScreen(SpriteBatch spriteBatch)
        {
            Rectangle screenDrawRect;
            Player player = gameState.Player;
            WorldMapCell worldMapCell = gameState.GetActiveWorldMapCell();

            camera.WorldMapCell = worldMapCell;
            camera.Position = player.Position;
            camera.Resolution = this.Location;

            Rectangle drawBounds = camera.GetDrawBounds();

            int dx = drawBounds.X;
            int dy = drawBounds.Y;

            //parallax
            if (worldMapCell.ParallaxBackground != null) DrawRepeatableParallax(spriteBatch, worldMapCell, player, drawBounds);

            //tiles
            for (int x = 0; x < worldMapCell.GetMaxTileX(); x++)
            {
                for (int y = 0; y < worldMapCell.GetMaxTileY(); y++)
                {
                    //layer 1
                    if (worldMapCell.TileSetsLayer1[x][y] != null)
                    {
                        screenDrawRect = new Rectangle((x * TileSet.DRAW_SIZE) - dx, (y * TileSet.DRAW_SIZE) - dy, TileSet.DRAW_SIZE, TileSet.DRAW_SIZE);
                        spriteBatch.Draw(worldMapCell.TileSetsLayer1[x][y].GetActiveTexture(), screenDrawRect, Color.White);
                    }

                    //layer 2 - not passable
                    if (worldMapCell.TileSetsLayer2[x][y] != null)
                    {
                        if (worldMapCell.TileSetsLayer2[x][y].Passable == false)
                        {
                            screenDrawRect = new Rectangle((x * TileSet.DRAW_SIZE) - dx, (y * TileSet.DRAW_SIZE) - dy, TileSet.DRAW_SIZE, TileSet.DRAW_SIZE);
                            spriteBatch.Draw(worldMapCell.TileSetsLayer2[x][y].GetActiveTexture(), screenDrawRect, Color.White);
                        }
                    }
                }
            }

            //items
            for(int i = 0; i < gameState.ActiveItems.Count; i++) gameState.ActiveItems[i].Draw(spriteBatch, dx, dy);

            //destructable blocks
            for (int i = 0; i < gameState.ActiveDestructableBlocks.Count; i++) gameState.ActiveDestructableBlocks[i].Draw(spriteBatch, dx, dy);
            
            //player
            player.Draw(spriteBatch, dx, dy);

            //enemies 
            for (int i = 0; i < gameState.ActiveEnemies.Count; i++) gameState.ActiveEnemies[i].Draw(spriteBatch, dx, dy);

            //teleporters
            for (int i = 0; i < gameState.ActiveTeleporters.Count; i++) gameState.ActiveTeleporters[i].Draw(spriteBatch, dx, dy);

            //savers
            for (int i = 0; i < gameState.ActiveSavers.Count; i++) gameState.ActiveSavers[i].Draw(spriteBatch, dx, dy);

            //victories
            for (int i = 0; i < gameState.ActiveVictories.Count; i++) gameState.ActiveVictories[i].Draw(spriteBatch, dx, dy);

            //weapon sprites
            for (int i = 0; i < gameState.ActiveWeaponSprites.Count; i++) gameState.ActiveWeaponSprites[i].Draw(spriteBatch, dx, dy);

            //doors
            for (int i = 0; i < gameState.ActiveDoors.Count; i++) gameState.ActiveDoors[i].Draw(spriteBatch, dx, dy);

            //tiles layer 2 - passable
            for (int x = 0; x < worldMapCell.GetMaxTileX(); x++)
            {
                for (int y = 0; y < worldMapCell.GetMaxTileY(); y++)
                {
                    if (worldMapCell.TileSetsLayer2[x][y] != null)
                    {
                        if (worldMapCell.TileSetsLayer2[x][y].Passable == true)
                        {
                            screenDrawRect = new Rectangle((x * TileSet.DRAW_SIZE) - dx, (y * TileSet.DRAW_SIZE) - dy, TileSet.DRAW_SIZE, TileSet.DRAW_SIZE);
                            spriteBatch.Draw(worldMapCell.TileSetsLayer2[x][y].GetActiveTexture(), screenDrawRect, Color.White);
                        }
                    }
                }
            }

            //effects
            for (int i = 0; i < gameState.ActiveEffects.Count; i++) gameState.ActiveEffects[i].Draw(spriteBatch, dx, dy);
        }


        private void DrawRepeatableParallax(SpriteBatch spriteBatch, WorldMapCell worldMapCell, Player player, Rectangle drawBounds)
        {
            Texture2D background = worldMapCell.ParallaxBackground;
            Rectangle screenDrawRect;

            int mw = worldMapCell.GetMaxTileX() * TileSet.DRAW_SIZE;
            int mh = worldMapCell.GetMaxTileY() * TileSet.DRAW_SIZE;

            bool cx = (mw == background.Width);
            bool cy = (mh == background.Height);

            int dx = -drawBounds.X;
            int dy = -drawBounds.Y;

            if (cx == false) dx = dx / 4;
            if (cy == false) dy = dy / 4;


            for(int i = 0; i < worldMapCell.WorldMapWidth; i++)
            {
                for(int j = 0; j < worldMapCell.WorldMapHeight; j++)
                {
                    screenDrawRect = new Rectangle(dx + (i*worldMapCell.ParallaxBackground.Width), dy + (j*worldMapCell.ParallaxBackground.Height), worldMapCell.ParallaxBackground.Width, worldMapCell.ParallaxBackground.Height);
                    spriteBatch.Draw(worldMapCell.ParallaxBackground, screenDrawRect, Color.White);
                }
            }
        }


        public void Press()
        {

        }


        public void Update(long ticks)
        {
            gameState.Update(ticks);

            for (int i = 0; i < this.Controls.Count; i++)
            {
                IControl control = this.Controls[i];
                control.Update(ticks);
            }

            CheckForMusicChange();
        }


        private void CheckForMusicChange()
        {
            WorldMapCell mapCell = this.gameState.GetActiveWorldMapCell();
            Song song = mapCell.Song;
            
            if (this.gameState.IsBossInRoom())
            {
                if (bossSong.Equals(this.screenController.BackgroundMusic) == false)
                {
                    MediaPlayer.Stop();
                    MediaPlayer.IsRepeating = true;
                    MediaPlayer.Play(bossSong);

                    this.screenController.BackgroundMusic = bossSong;
                }
            }
            else
            {
                if (song.Equals(this.screenController.BackgroundMusic) == false)
                {
                    MediaPlayer.Stop();
                    MediaPlayer.IsRepeating = true;
                    MediaPlayer.Play(song);

                    this.screenController.BackgroundMusic = song;
                }
            }
        }


    }
}
