﻿using GameResources.Models.Samus;
using GameResources.Repository;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using System;
using System.Collections.Generic;


namespace UserInterface.Control
{
    public class PlayerHud : IControl
    {
        private readonly IGameNamedContentRepository<Texture2D> textureRepository;
        private readonly IGameNamedContentRepository<SpriteFont> fontRepository;

        public IList<IControl> Controls { get; set; }
        public bool IsSelected { get; set; }
        public bool IsSelectable { get; set; }
        public Rectangle Location { get; set; }
        public Player Player { get; set; }

        private readonly Texture2D healthFull;
        private readonly Texture2D healthEmpty;
        private readonly Texture2D background;
        private readonly SpriteFont font;
        private readonly Texture2D missileIcon;
        private readonly Texture2D missileIconSelected;
        private readonly Texture2D superMissileIcon;
        private readonly Texture2D superMissileIconSelected;
        private readonly Texture2D superBombIcon;
        private readonly Texture2D superBombIconSelected;

        private int leftOverHealth;
        private string leftOverHealthString;


        public PlayerHud(IGameNamedContentRepository<Texture2D> textureRepository, IGameNamedContentRepository<SpriteFont> fontRepository)
        {
            this.textureRepository = textureRepository;
            this.fontRepository = fontRepository;
            this.Controls = new List<IControl>();

            healthFull = this.textureRepository.GetContentByName(@"textures\ui\health-full");
            healthEmpty = this.textureRepository.GetContentByName(@"textures\ui\health-empty");
            background = this.textureRepository.GetContentByName(@"textures\white");

            missileIcon = this.textureRepository.GetContentByName(@"textures\ui\missile-indicator-inactive");
            missileIconSelected = this.textureRepository.GetContentByName(@"textures\ui\missile-indicator-active");
            superMissileIcon = this.textureRepository.GetContentByName(@"textures\ui\supermissile-indicator-inactive");
            superMissileIconSelected = this.textureRepository.GetContentByName(@"textures\ui\supermissile-indicator-active");
            superBombIcon = this.textureRepository.GetContentByName(@"textures\ui\superbomb-indicator-inactive");
            superBombIconSelected = this.textureRepository.GetContentByName(@"textures\ui\superbomb-indicator-active");

            font = this.fontRepository.GetContentByName(@"fonts\Normal");

            this.leftOverHealth = 0;
            this.leftOverHealthString = "";
        }


        public void Update(long ticks)
        {
            
        }


        public void Draw(SpriteBatch spriteBatch)
        {
            if (this.Player == null) return;

            Rectangle healthArea = DrawHealth(spriteBatch);
            Rectangle weaponArea = DrawWeapons(spriteBatch, healthArea);
        }


        private Rectangle DrawHealth(SpriteBatch spriteBatch)
        {
            int numContainers = this.Player.CharacterStats.MaximumHealth / 100;
            int numFullContainers = this.Player.CharacterStats.Health / 100;
            int leftOver = this.Player.CharacterStats.Health - (numFullContainers * 100);

            if (leftOver != this.leftOverHealth)
            {
                this.leftOverHealth = leftOver;
                this.leftOverHealthString = leftOver.ToString();
            }

            int padding = 4;
            int containerWidth = 16;

            int col = 0;
            int row = 1;
            int containersPerRow = 6;
            int numRows = 1;

            //background
            Rectangle backgroundRect = new Rectangle(this.Location.X, this.Location.Y, (containersPerRow + 2) * (padding + containerWidth), (numRows* containerWidth) + ((numRows+1) * padding));
            spriteBatch.Draw(background, backgroundRect, Color.Black);

            //containers
            for (int i = 1; i <= numContainers; i++)
            {
                Texture2D texture = (i <= numFullContainers) ? this.healthFull : this.healthEmpty;

                if ((i % (containersPerRow + 1) == 0))
                {
                    col = 1;
                    row++;
                }
                else
                {
                    col++;
                }

                int x = this.Location.X + padding + ((col-1) * (containerWidth + padding));
                int y = this.Location.Y + padding + ((row-1) * (containerWidth + padding));

                spriteBatch.Draw(texture, new Rectangle(x, y, containerWidth, containerWidth), Color.White);
            }

            //text
            Vector2 position = new Vector2(this.Location.X + padding + (containersPerRow * (containerWidth + padding)), this.Location.Y + padding + ((numRows-1) * 2 * padding));
            spriteBatch.DrawString(font, this.leftOverHealthString, position, Color.White);

            return backgroundRect;
        }


        private Rectangle DrawWeapons(SpriteBatch spriteBatch, Rectangle previousArea)
        {
            int spacingFromPrevious = 16;
            int xPadding = 8;
            int yPadding = 4;
            int spacing = 16;
            int iconSize = 32;
            int numIcons = 3;
            int textHeight = 20;

            Rectangle rect = default;
            int iconIndex = 0;
            int iconX = 0;
            Texture2D texture = null;
            PlayerInventory.Weapons selectedWeapon = Player.Inventory.SelectedWeapon;

            int startX = previousArea.X + previousArea.Width + spacingFromPrevious;
            int fontY = this.Location.Y + iconSize + yPadding;

            //background
            Rectangle backgroundRect = new Rectangle(startX, this.Location.Y, (numIcons*iconSize) + ((numIcons-1)*spacing) + (xPadding * 2), iconSize + textHeight + (yPadding * 2));
            spriteBatch.Draw(background, backgroundRect, Color.Black);

            //missiles
            iconIndex = 0;
            iconX = startX + (iconIndex * iconSize) + (iconIndex * spacing) + xPadding;
            texture = (selectedWeapon == PlayerInventory.Weapons.Missile) ? this.missileIconSelected : this.missileIcon;
            rect = new Rectangle(iconX, this.Location.Y + yPadding, iconSize, iconSize);
            spriteBatch.Draw(texture, rect, Color.White);
            spriteBatch.DrawString(font, Player.Inventory.Missiles.ToString(), new Vector2(iconX, fontY), Color.White);

            //super missiles
            iconIndex = 1;
            iconX = startX + (iconIndex * iconSize) + (iconIndex * spacing) + xPadding;
            texture = (selectedWeapon == PlayerInventory.Weapons.SuperMissle) ? this.superMissileIconSelected : this.superMissileIcon;
            rect = new Rectangle(iconX, this.Location.Y + yPadding, iconSize, iconSize);
            spriteBatch.Draw(texture, rect, Color.White);
            spriteBatch.DrawString(font, Player.Inventory.SuperMissiles.ToString(), new Vector2(iconX, fontY), Color.White);

            //super bombs
            iconIndex = 2;
            iconX = startX + (iconIndex * iconSize) + (iconIndex * spacing) + xPadding;
            texture = (selectedWeapon == PlayerInventory.Weapons.SuperBomb) ? this.superBombIconSelected : this.superBombIcon;
            rect = new Rectangle(iconX, this.Location.Y + yPadding, iconSize, iconSize);
            spriteBatch.Draw(texture, rect, Color.White);
            spriteBatch.DrawString(font, Player.Inventory.SuperBombs.ToString(), new Vector2(iconX, fontY), Color.White);

            return backgroundRect;
        }


        public void Press()
        {
            
        }


    }
}
