﻿using GameResources.Manager;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using RawResources.Manager;
using System;
using System.Collections.Generic;
using UserInterface.Core;
using UserInterface.Screen;
using GameResources.Models.Game;
using System.Reflection;
using System.IO;
using GameResources;

namespace Metroid
{
    public class GameEngine: Game, IScreenController
    {
        public GraphicsDeviceManager graphics;
        public IResolution Resolution { get; }
        public SpriteBatch SpriteBatch { get; private set; }
        public Song BackgroundMusic { get; set; }
        public long LastKeyboardInputTicks { get; set; }
        
        private long lastResolutionInputTicks;
        private bool fullScreen;
        private IDictionary<ScreenType, IScreen> screens;
        private ScreenType activeScreen;
        
        public IGameContentManager GameContentManager { get; set; }
        public GameState GameState { get; set; }




        public GameEngine()
        {
            fullScreen = false;

            Content.RootDirectory = "Content/bin";

            this.Window.Title = "Metroid";

            MediaPlayer.Volume = 0.2f; 
            SoundEffect.MasterVolume = 0.2f;

            this.IsFixedTimeStep = true;
            this.TargetElapsedTime = TimeSpan.FromTicks(166666);
            this.Window.AllowUserResizing = true;

            graphics = new GraphicsDeviceManager(this);

            this.Resolution = new Resolution(this.graphics)
            {
                GameWidth = UserInterface.Core.Resolution.RESOLUTION_X,
                GameHeight = UserInterface.Core.Resolution.RESOLUTION_Y
            };

            SetupGraphics();

            this.Window.ClientSizeChanged += Window_ClientSizeChanged;
        }


        private void SetupGraphics()
        {
            graphics.IsFullScreen = fullScreen;
            
            if (fullScreen)
            {
                graphics.PreferredBackBufferWidth = 1920;
                graphics.PreferredBackBufferHeight = 1080;
            }
            else
            {
                graphics.PreferredBackBufferWidth = UserInterface.Core.Resolution.RESOLUTION_X;
                graphics.PreferredBackBufferHeight = UserInterface.Core.Resolution.RESOLUTION_Y;
            }

            graphics.ApplyChanges();
        }


        private void Window_ClientSizeChanged(object sender, EventArgs e)
        {
            ChangeResolution();
        }


        protected override void Initialize()
        {
            base.Initialize();

            this.screens = new Dictionary<ScreenType, IScreen>
            {
                { ScreenType.MainMenu, new MainMenuScreen(this) },
                { ScreenType.Game, new GameScreen(this) },
                { ScreenType.GameMenu, new GameMenuScreen(this) },
                { ScreenType.GameOver, new GameOverScreen(this) },
                { ScreenType.Story, new StoryScreen(this) },
                { ScreenType.Victory, new VictoryScreen(this) }
            };

            ChangeScreen(ScreenType.MainMenu);
        }


        protected override void LoadContent()
        {
            this.SpriteBatch = new SpriteBatch(GraphicsDevice);

            string directory = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + Path.DirectorySeparatorChar;
            IRawContentManager rawContentManager = new DefaultRawContentManager(directory);
            this.GameContentManager = new DefaultGameContentManager(this.Content, rawContentManager);

            SetupGraphics();

            ChangeResolution();
        }


        protected override void UnloadContent()
        {
            base.UnloadContent();
        }


        protected override void Update(GameTime gameTime)
        {
            base.Update(gameTime);
            
            long ticks = gameTime.ElapsedGameTime.Ticks;


                this.lastResolutionInputTicks += ticks;
                this.LastKeyboardInputTicks += ticks;

                if (GetInputMap().ScreenToggle)
                {
                    if (this.lastResolutionInputTicks > TimeSpan.TicksPerSecond)
                    {
                        this.Window.ClientSizeChanged -= Window_ClientSizeChanged;

                        fullScreen = !fullScreen;
                        SetupGraphics();
                        ChangeResolution();

                        this.Window.ClientSizeChanged += Window_ClientSizeChanged;

                        this.lastResolutionInputTicks = 0;
                        this.LastKeyboardInputTicks = 0;
                    }
                }

                this.screens[this.activeScreen].Update(ticks);


            //System.Console.Out.WriteLine(ticks);
            //double fps = 1 / ((double)ticks / TimeSpan.TicksPerSecond);
            //System.Console.Out.WriteLine(fps.ToString("#"));

            //if(gameTime.IsRunningSlowly) System.Console.Out.WriteLine("TRUE" + gameTime.TotalGameTime);
        }


        protected override void Draw(GameTime gameTime)
        {
            this.graphics.GraphicsDevice.Clear(Color.Black);

            this.SpriteBatch.Begin(transformMatrix: this.Resolution.TransformationMatrix, samplerState: SamplerState.PointClamp);

            this.screens[this.activeScreen].Draw(this.SpriteBatch);

            SpriteBatch.End();

            base.Draw(gameTime);
        }


        private void ChangeResolution()
        {
            if (this.Resolution != null)
            {
                this.Resolution.ActualWidth = this.Window.ClientBounds.Width;
                this.Resolution.ActualHeight = this.Window.ClientBounds.Height;
                this.Resolution.Update();
            }
        }


        public void ChangeScreen(ScreenType screenType)
        {
            this.activeScreen = screenType;

            this.screens[this.activeScreen].ActivateScreen();
        }


        public InputMap GetInputMap()
        {
            InputMap input = new InputMap()
            {
                Up = Keyboard.GetState().IsKeyDown(Keys.Up) || GamePad.GetState(0).IsButtonDown(Buttons.DPadUp) || GamePad.GetState(0).IsButtonDown(Buttons.LeftThumbstickUp),
                Down = Keyboard.GetState().IsKeyDown(Keys.Down) || GamePad.GetState(0).IsButtonDown(Buttons.DPadDown) || GamePad.GetState(0).IsButtonDown(Buttons.LeftThumbstickDown),
                Left = Keyboard.GetState().IsKeyDown(Keys.Left) || GamePad.GetState(0).IsButtonDown(Buttons.DPadLeft) || GamePad.GetState(0).IsButtonDown(Buttons.LeftThumbstickLeft),
                Right = Keyboard.GetState().IsKeyDown(Keys.Right) || GamePad.GetState(0).IsButtonDown(Buttons.DPadRight) || GamePad.GetState(0).IsButtonDown(Buttons.LeftThumbstickRight),
                Jump = Keyboard.GetState().IsKeyDown(Keys.X) || GamePad.GetState(0).IsButtonDown(Buttons.A),
                Shoot = Keyboard.GetState().IsKeyDown(Keys.W) || GamePad.GetState(0).IsButtonDown(Buttons.X),
                AngleUp = Keyboard.GetState().IsKeyDown(Keys.Z) || GamePad.GetState(0).IsButtonDown(Buttons.LeftShoulder),
                AngleDown = Keyboard.GetState().IsKeyDown(Keys.C) || GamePad.GetState(0).IsButtonDown(Buttons.LeftTrigger),
                ChangeItem = Keyboard.GetState().IsKeyDown(Keys.A) || GamePad.GetState(0).IsButtonDown(Buttons.B),
                Escape = Keyboard.GetState().IsKeyDown(Keys.Escape) || GamePad.GetState(0).IsButtonDown(Buttons.Start),
                ScreenToggle = Keyboard.GetState().IsKeyDown(Keys.OemTilde) || GamePad.GetState(0).IsButtonDown(Buttons.Back)
            };

            return input;
        }


    }
}
