﻿using GameResources.Manager;
using GameResources.Models.Game;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Media;
using System;
using UserInterface.Control;


namespace UserInterface.Screen
{
    public class GameOverScreen : IScreen
    {
        public ControlCollection Controls { get; }
        private readonly IScreenController screenController;

        private long lastMovementInput;
        private long lastActionInput;


        public GameOverScreen(IScreenController screenController)
        {
            this.Controls = new ControlCollection();
            this.screenController = screenController;

            Init();
        }


        public void Draw(SpriteBatch spriteBatch)
        {
            for (int i = 0; i < this.Controls.Count; i++)
            {
                IControl control = this.Controls[i];
                control.Draw(spriteBatch);
            }
        }


        public void Update(long ticks)
        {
            InputMap inputMap = screenController.GetInputMap();

            lastMovementInput += ticks;
            lastActionInput += ticks;

            //keyboard
            if (lastMovementInput >= (TimeSpan.TicksPerSecond * 0.12))
            {
                if (inputMap.Up)
                {
                    this.Controls.SelectPrev();
                    lastMovementInput = 0;
                }
                else if (inputMap.Down)
                {
                    this.Controls.SelectNext();
                    lastMovementInput = 0;
                }
            }

            if (lastActionInput >= (TimeSpan.TicksPerSecond * 0.12))
            {
                if (inputMap.Jump)
                {
                    if (this.Controls.IndexOfSelected() != -1)
                    {
                        lastActionInput = 0;
                        screenController.LastKeyboardInputTicks = 0;

                        this.Controls[this.Controls.IndexOfSelected()].Press();
                    }
                }
            }

            //controls
            foreach (IControl control in this.Controls)
            {
                control.Update(ticks);
            }
        }


        public void ActivateScreen()
        {
            Song song = this.screenController.GameContentManager.SongRepository.GetContentByName(@"music\theme");

            if (MediaPlayer.State == MediaState.Playing)
            {
                MediaPlayer.Stop();
            }

            this.screenController.BackgroundMusic = song;
            MediaPlayer.IsRepeating = true;
            MediaPlayer.Play(song);

            lastMovementInput = screenController.LastKeyboardInputTicks;
            lastActionInput = screenController.LastKeyboardInputTicks;
        }


        private void Init()
        {
            IGameContentManager gameContentManager = this.screenController.GameContentManager;

            int buttonWidth = 220;
            int buttonHeight = 32;
            int buttonStart = -32;
            int buttonSpace = 48;
            int screenWidth = screenController.Resolution.GameWidth;
            int screenHeight = screenController.Resolution.GameHeight;
            int screenHeightHalf = screenHeight / 2;

            this.Controls.Add(new Label()
            {
                Text = "Game Over",
                Font = gameContentManager.FontRepository.GetContentByName(@"fonts\Title"),
                Location = new Rectangle((screenWidth - 400) / 2, 40, 400, 100)
            });

            this.Controls.Add(new Button()
            {
                Text = "Main Menu",
                Location = new Rectangle((screenWidth - buttonWidth) / 2, screenHeightHalf + buttonStart + (buttonSpace * 1), buttonWidth, buttonHeight),
                TextureNormal = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button"),
                TextureSelected = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-select"),
                TexturePressed = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-press"),
                Font = gameContentManager.FontRepository.GetContentByName(@"fonts\Normal"),
                PressSound = gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Samus Land"),
                IsSelected = true,
                CommandText = "2"
            });
            ((Button)this.Controls[this.Controls.Count - 1]).OnClick += MainMenu;
        }


        private void MainMenu(object sender)
        {
            this.screenController.GameState = null;

            this.screenController.ChangeScreen(ScreenType.MainMenu);
        }


    }
}
