﻿using Avalonia;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media.Imaging;
using GameMakerAvalonia.Dialogs;
using GameMakerAvalonia.Models;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;

namespace GameMakerAvalonia.Panels
{
    public class EnemiesPanel : UserControl
    {
        private Session session;
        private IList<EnemyDefinition> resources;


        public EnemiesPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);
            
            AddEvents();

            this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
        }


        private void AddEvents()
        {
            this.FindControl<Button>("AddBlock").Click += AddBlock_Click;
            this.FindControl<Button>("DeleteBlock").Click += DeleteBlock_Click;
            this.FindControl<ComboBox>("MotionType").SelectionChanged += MotionType_SelectedChanged;
            this.FindControl<TextBox>("Name").PropertyChanged += Name_TextChanged;
            this.FindControl<TextBox>("Health").PropertyChanged += Health_TextChanged;
            this.FindControl<TextBox>("Attack").PropertyChanged += Attack_TextChanged;
            this.FindControl<TextBox>("ItemDropChance").PropertyChanged += ItemDropChance_TextChanged;
            this.FindControl<TextBox>("OnlyShowIfVariableIsFalse").PropertyChanged += OnlyShowIfVariableIsFalse_TextChanged;
            this.FindControl<TextBox>("SetVariableToTrueUponDeath").PropertyChanged += SetVariableToTrueUponDeath_TextChanged;
            this.FindControl<CheckBox>("IsScrewable").Click += IsScrewable_Checked;
            this.FindControl<CheckBox>("IsBombable").Click += IsBombable_Checked;
            this.FindControl<CheckBox>("IsFreezeable").Click += IsFreezeable_Checked;
            this.FindControl<CheckBox>("IsImmediatelyFreezeable").Click += IsImmediatelyFreezable_Checked;
            this.FindControl<CheckBox>("IsBoss").Click += IsBoss_Checked;
            this.FindControl<CheckBox>("IsExplosive").Click += IsExplosive_Checked;
            this.FindControl<Button>("Texture1SelectButton").Click += Texture1SelectButton_Click;
            this.FindControl<Button>("Texture2SelectButton").Click += Texture2SelectButton_Click;

            this.FindControl<CheckBox>("CanShoot").Click += CanShoot_Checked;
            this.FindControl<TextBox>("ShootSpeed").PropertyChanged += ShootSpeed_TextChanged;
            this.FindControl<ComboBox>("EnemyForShooting").SelectionChanged += EnemyForShooting_SelectedChanged;
        }


        private void RemoveEvents()
        {
            this.FindControl<Button>("AddBlock").Click -= AddBlock_Click;
            this.FindControl<Button>("DeleteBlock").Click -= DeleteBlock_Click;
            this.FindControl<ComboBox>("MotionType").SelectionChanged -= MotionType_SelectedChanged;
            this.FindControl<TextBox>("Name").PropertyChanged -= Name_TextChanged;
            this.FindControl<TextBox>("Health").PropertyChanged -= Health_TextChanged;
            this.FindControl<TextBox>("Attack").PropertyChanged -= Attack_TextChanged;
            this.FindControl<TextBox>("ItemDropChance").PropertyChanged -= ItemDropChance_TextChanged;
            this.FindControl<TextBox>("OnlyShowIfVariableIsFalse").PropertyChanged -= OnlyShowIfVariableIsFalse_TextChanged;
            this.FindControl<TextBox>("SetVariableToTrueUponDeath").PropertyChanged -= SetVariableToTrueUponDeath_TextChanged;
            this.FindControl<CheckBox>("IsScrewable").Click -= IsScrewable_Checked;
            this.FindControl<CheckBox>("IsBombable").Click -= IsBombable_Checked;
            this.FindControl<CheckBox>("IsFreezeable").Click -= IsFreezeable_Checked;
            this.FindControl<CheckBox>("IsImmediatelyFreezeable").Click -= IsImmediatelyFreezable_Checked;
            this.FindControl<CheckBox>("IsBoss").Click -= IsBoss_Checked;
            this.FindControl<CheckBox>("IsExplosive").Click -= IsExplosive_Checked;
            this.FindControl<Button>("Texture1SelectButton").Click -= Texture1SelectButton_Click;
            this.FindControl<Button>("Texture2SelectButton").Click -= Texture2SelectButton_Click;

            this.FindControl<CheckBox>("CanShoot").Click -= CanShoot_Checked;
            this.FindControl<TextBox>("ShootSpeed").PropertyChanged -= ShootSpeed_TextChanged;
            this.FindControl<ComboBox>("EnemyForShooting").SelectionChanged -= EnemyForShooting_SelectedChanged;
        }
        

        public void SetSession(Session session)
        {
            this.session = session;

            LoadSession();
        }


        private void LoadSession()
        {
            this.resources = session.RawContentManager.EnemyRepository.GetListOfContent();

            var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;

            RefreshForm();
        }


        private void LoadComboBoxes()
        {
            ComboBox motionType = this.FindControl<ComboBox>("MotionType");
            motionType.Items = Enum.GetValues(typeof(EnemyMotionType));

            ComboBox enemyForShooting = this.FindControl<ComboBox>("EnemyForShooting");
            IList<EnemyDefinition> enemies = session.RawContentManager.EnemyRepository.GetListOfContent();
            enemies.Insert(0, null);
            enemyForShooting.Items = enemies;
        }


        private void ListGridView_CurrentCellChanged(object sender, EventArgs e)
        {
            RefreshForm();
        }


        private void RefreshForm()
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            ComboBox motionType = this.FindControl<ComboBox>("MotionType");
            TextBox name = this.FindControl<TextBox>("Name");
            TextBox health = this.FindControl<TextBox>("Health");
            TextBox attack = this.FindControl<TextBox>("Attack");
            TextBox itemDropChance = this.FindControl<TextBox>("ItemDropChance");
            TextBox onlyShowIfVariableIsFalse = this.FindControl<TextBox>("OnlyShowIfVariableIsFalse");
            TextBox setVariableToTrueUponDeath = this.FindControl<TextBox>("SetVariableToTrueUponDeath");
            CheckBox isScrewable = this.FindControl<CheckBox>("IsScrewable");
            Image texture1Preview = this.FindControl<Image>("Texture1Preview");
            Image texture2Preview = this.FindControl<Image>("Texture2Preview");
            CheckBox isBombable = this.FindControl<CheckBox>("IsBombable");
            CheckBox isFreezeable = this.FindControl<CheckBox>("IsFreezeable");
            CheckBox isImmediatelyFreezeable = this.FindControl<CheckBox>("IsImmediatelyFreezeable");
            CheckBox isBoss = this.FindControl<CheckBox>("IsBoss");
            CheckBox isExplosive = this.FindControl<CheckBox>("IsExplosive");

            CheckBox canShoot = this.FindControl<CheckBox>("CanShoot");
            TextBox shootSpeed = this.FindControl<TextBox>("ShootSpeed");
            ComboBox enemyForShooting = this.FindControl<ComboBox>("EnemyForShooting");

            RemoveEvents();

            LoadComboBoxes();

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                motionType.SelectedItem = enemy.MotionType;
                name.Text = enemy.Name;
                health.Text = enemy.Health.ToString();
                attack.Text = enemy.Attack.ToString();
                itemDropChance.Text = enemy.ItemDropChance.ToString();
                onlyShowIfVariableIsFalse.Text = enemy.OnlyShowIfVariableIsFalse;
                setVariableToTrueUponDeath.Text = enemy.SetVariableToTrueUponDeath;
                isScrewable.IsChecked = enemy.IsScrewable;
                isBombable.IsChecked = enemy.IsBombable;
                isFreezeable.IsChecked = enemy.IsFreezeable;
                isImmediatelyFreezeable.IsChecked = enemy.IsImmediatelyFreezable;
                isBoss.IsChecked = enemy.IsBoss;
                isExplosive.IsChecked = enemy.IsExplosive;
                canShoot.IsChecked = enemy.CanShoot;
                shootSpeed.Text = enemy.ShootSpeed.ToString();

                if (enemy.EnemyKeyForShooting == null) enemyForShooting.SelectedItem = null;
                else enemyForShooting.SelectedItem = session.RawContentManager.EnemyRepository.GetContent(enemy.EnemyKeyForShooting.Value);
                
                LoadTexture(enemy.TextureKey1, texture1Preview);
                LoadTexture(enemy.TextureKey2, texture2Preview);
            }
            else
            {
                motionType.SelectedItem = null;
                name.Text = "";
                health.Text = "";
                attack.Text = "";
                itemDropChance.Text = "";
                onlyShowIfVariableIsFalse.Text = "";
                setVariableToTrueUponDeath.Text = "";
                isScrewable.IsChecked = false;
                isBombable.IsChecked = false;
                isFreezeable.IsChecked = false;
                isImmediatelyFreezeable.IsChecked = false;
                isBoss.IsChecked = false;
                isExplosive.IsChecked = false;
                canShoot.IsChecked = false;
                shootSpeed.Text = "";
                enemyForShooting.SelectedItem = null;
                texture1Preview.Source = null;
                texture2Preview.Source = null;
            }

            AddEvents();
        }


        private void AddBlock_Click(object sender, RoutedEventArgs e)
        {
            EnemyDefinition enemy = new EnemyDefinition();

            this.session.RawContentManager.EnemyRepository.AddContent(enemy);

            var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = null;

            LoadSession();
        }


        private async void DeleteBlock_Click(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Select an enemy",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);

                return;
            }

            EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

            this.session.RawContentManager.EnemyRepository.RemoveContent(enemy.Key);

            grid.Items = null;

            LoadSession();
        }


        private async void Texture1SelectButton_Click(object sender, RoutedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                TextureSelectDialog dialog = new TextureSelectDialog();
                dialog.SetParameters(session, enemy.TextureKey1);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == ButtonResult.Ok)
                {
                    enemy.TextureKey1 = dialog.SelectedTexture.Key;

                    RefreshForm();
                }
            }
        }


        private async void Texture2SelectButton_Click(object sender, RoutedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                TextureSelectDialog dialog = new TextureSelectDialog();
                dialog.SetParameters(session, enemy.TextureKey2);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == ButtonResult.Ok)
                {
                    enemy.TextureKey2 = dialog.SelectedTexture.Key;

                    RefreshForm();
                }
            }
        }


        private void LoadTexture(int textureKey, Image image)
        {
            RawResource texture = this.session.RawContentManager.TextureRepository.GetContent(textureKey);
            string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + texture.SystemFilePath + ".png";
            image.Source = new Bitmap(filename);
        }


        private void MotionType_SelectedChanged(object sender, SelectionChangedEventArgs e)
        {
            ComboBox motionType = this.FindControl<ComboBox>("MotionType");
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                enemy.MotionType = (EnemyMotionType)motionType.SelectedItem;
            }
        }


        private void Name_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                    TextBox text = this.FindControl<TextBox>("Name");

                    enemy.Name = text.Text;
                }
            }
        }


        private void Health_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                    TextBox text = this.FindControl<TextBox>("Health");

                    enemy.Health = text.Text.Equals("") ? 0 : Int32.Parse(text.Text);
                }
            }
        }


        private void Attack_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                    TextBox text = this.FindControl<TextBox>("Attack");

                    enemy.Attack = text.Text.Equals("") ? 0 : Int32.Parse(text.Text);
                }
            }
        }


        private void ItemDropChance_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                    TextBox text = this.FindControl<TextBox>("ItemDropChance");

                    enemy.ItemDropChance = text.Text.Equals("") ? 0 : Double.Parse(text.Text);
                }
            }
        }


        private void OnlyShowIfVariableIsFalse_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                    TextBox text = this.FindControl<TextBox>("OnlyShowIfVariableIsFalse");

                    enemy.OnlyShowIfVariableIsFalse = text.Text;
                }
            }
        }


        private void SetVariableToTrueUponDeath_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                    TextBox text = this.FindControl<TextBox>("SetVariableToTrueUponDeath");

                    enemy.SetVariableToTrueUponDeath = text.Text;
                }
            }
        }


        private void IsScrewable_Checked(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                CheckBox checkBox = this.FindControl<CheckBox>("IsScrewable");

                enemy.IsScrewable = checkBox.IsChecked.Value;
            }
        }


        private void IsBombable_Checked(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                CheckBox checkBox = this.FindControl<CheckBox>("IsBombable");

                enemy.IsBombable = checkBox.IsChecked.Value;
            }
        }


        private void IsFreezeable_Checked(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                CheckBox checkBox = this.FindControl<CheckBox>("IsFreezeable");

                enemy.IsFreezeable = checkBox.IsChecked.Value;
            }
        }


        private void IsImmediatelyFreezable_Checked(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                CheckBox checkBox = this.FindControl<CheckBox>("IsImmediatelyFreezeable");

                enemy.IsImmediatelyFreezable = checkBox.IsChecked.Value;
            }
        }


        private void IsBoss_Checked(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                CheckBox checkBox = this.FindControl<CheckBox>("IsBoss");

                enemy.IsBoss = checkBox.IsChecked.Value;
            }
        }


        private void IsExplosive_Checked(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                CheckBox checkBox = this.FindControl<CheckBox>("IsExplosive");

                enemy.IsExplosive = checkBox.IsChecked.Value;
            }
        }


        private void CanShoot_Checked(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                CheckBox checkBox = this.FindControl<CheckBox>("CanShoot");

                enemy.CanShoot = checkBox.IsChecked.Value;
            }
        }


        private void ShootSpeed_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;
                    TextBox text = this.FindControl<TextBox>("ShootSpeed");

                    enemy.ShootSpeed = text.Text.Equals("") ? 0 : Double.Parse(text.Text);
                }
            }
        }


        private void EnemyForShooting_SelectedChanged(object sender, SelectionChangedEventArgs e)
        {
            ComboBox enemyForShooting = this.FindControl<ComboBox>("EnemyForShooting");
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                EnemyDefinition shootable = (EnemyDefinition)enemyForShooting.SelectedItem;

                if (shootable == null) enemyForShooting = null;
                else enemy.EnemyKeyForShooting = shootable.Key;
            }
        }


    }
}
