﻿using Avalonia;
using Avalonia.Controls;
using Avalonia.Input;
using Avalonia.Markup.Xaml;
using Avalonia.Media;
using Avalonia.Media.Imaging;
using GameMakerAvalonia.Dialogs;
using GameMakerAvalonia.Logic;
using GameMakerAvalonia.Models;
using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;
using System.Threading.Tasks;


namespace GameMakerAvalonia.PreviewPanels
{
    public class WorldCellPreviewPanel : UserControl
    {
        public enum EditMode { ReadOnly, Tile, Door, DestructableBlock, Teleporter, Item, Saver, Enemy, Victory };

        public EditMode SelectedEditMode { get; set; }
        public TileSetPreviewPanel TileSetSelectionPanel { get; set; }
        public int? SelectedTileX { get; set; }
        public int? SelectedTileY { get; set; }

        private Session session;
        private WorldMapCellDefinition worldMapCell;
        private int tileSize;
        private IDictionary<int, IBitmap> cacheTileImages;
        private IDictionary<DoorDirection, IBitmap> cacheDoorOpenImages;
        private IDictionary<DoorDirection, IBitmap> cacheDoorCloseImages;
        private IDictionary<ItemType, IBitmap> cacheItemImages;
        private IDictionary<int, IBitmap> cacheEnemyImages;
        public bool IsShiftDown { get; set; }
        public bool IsControlDown { get; set; }


        public WorldCellPreviewPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.cacheTileImages = new Dictionary<int, IBitmap>();
            this.cacheDoorOpenImages = new Dictionary<DoorDirection, IBitmap>();
            this.cacheDoorCloseImages = new Dictionary<DoorDirection, IBitmap>();
            this.cacheItemImages = new Dictionary<ItemType, IBitmap>();
            this.cacheEnemyImages = new Dictionary<int, IBitmap>();
            this.TileSize = 64;
            this.SelectedEditMode = EditMode.ReadOnly;

            this.PointerPressed += WorldCellPreviewPanel_PointerPressed;
            this.PointerMoved += WorldCellPreviewPanel_PointerMoved;

            this.IsControlDown = false;
            this.IsShiftDown = false;
        }


        public int TileSize
        {
            get
            {
                return tileSize;
            }
            set
            {
                tileSize = value;

                if (this.worldMapCell != null)
                {
                    this.Width = tileSize * this.worldMapCell.GetMaxTileX();
                    this.Height = tileSize * this.worldMapCell.GetMaxTileY();
                }
                else
                {
                    this.Width = tileSize;
                    this.Height = tileSize;
                }
            }
        }


        public void SetSession(Session session)
        {
            this.session = session;

            this.TileSize = this.TileSize;
        }


        public void SetWorldMapCell(WorldMapCellDefinition worldMapCell)
        {
            this.worldMapCell = worldMapCell;

            this.TileSize = this.TileSize;
        }


        public override void Render(DrawingContext graphics)
        {
            base.Render(graphics);

            if (this.worldMapCell == null) return;
            if (this.worldMapCell.TileSetKeysLayer1.Count == 0) return;

            IBrush backgroundBrush = new SolidColorBrush(Color.FromRgb(Convert.ToByte(0), Convert.ToByte(0), Convert.ToByte(0)));
            IBrush defaultBrush = new SolidColorBrush(Color.FromArgb(Convert.ToByte(0), Convert.ToByte(255), Convert.ToByte(255), Convert.ToByte(255)));
            IBrush selectedPenBrush = new SolidColorBrush(Color.FromRgb(Convert.ToByte(255), Convert.ToByte(0), Convert.ToByte(0)));
            IPen selectedPen = new Pen(selectedPenBrush);
            IBrush doorLinePenBrush = new SolidColorBrush(Color.FromArgb(Convert.ToByte(255), Convert.ToByte(0), Convert.ToByte(255), Convert.ToByte(0)));
            IPen doorLinePen = new Pen(doorLinePenBrush);

            //Background
            graphics.FillRectangle(backgroundBrush, new Rect(0, 0, this.Width, this.Height));

            //Tiles & Selection
            foreach (int kx in this.worldMapCell.TileSetKeysLayer1.Keys)
            {
                foreach (int ky in this.worldMapCell.TileSetKeysLayer1[kx].Keys)
                {
                    int? tileKey = null;
                    Rect rect = new Rect(kx * TileSize, ky * TileSize, TileSize, TileSize);

                    //tile layer 1
                    tileKey = this.worldMapCell.TileSetKeysLayer1[kx][ky];
                    if (tileKey != null)
                    {
                        TileSetDefinition tile = this.session.RawContentManager.TileSetRepository.GetContent(tileKey.Value);
                        IBitmap image = GetTileImage(tile);

                        graphics.DrawImage(image, rect);
                    }
                    else
                    {
                        graphics.FillRectangle(defaultBrush, rect);
                    }

                    //tile layer 2
                    tileKey = this.worldMapCell.TileSetKeysLayer2[kx][ky];
                    if (tileKey != null)
                    {
                        TileSetDefinition tile = this.session.RawContentManager.TileSetRepository.GetContent(tileKey.Value);
                        IBitmap image = GetTileImage(tile);

                        graphics.DrawImage(image, rect);
                    }
                    else
                    {
                        graphics.FillRectangle(defaultBrush, rect);
                    }

                    //selection 
                    if (this.SelectedEditMode != EditMode.Tile)
                    {
                        if ((kx == this.SelectedTileX) && (ky == this.SelectedTileY))
                        {
                            rect = new Rect(kx * TileSize, ky * TileSize, TileSize - 1, TileSize - 1);
                            graphics.DrawRectangle(selectedPen, rect);
                        }
                    }
                }
            }

            //Cell Half-Point Markers
            int halfX = (this.worldMapCell.TileSetKeysLayer1.Keys.Count / 2) / this.worldMapCell.WorldMapWidth * tileSize;
            int halfY = (this.worldMapCell.TileSetKeysLayer1[0].Keys.Count / 2) / this.worldMapCell.WorldMapHeight * tileSize;

            int maxX = this.worldMapCell.TileSetKeysLayer1.Keys.Count * TileSize;
            int maxY = this.worldMapCell.TileSetKeysLayer1[0].Keys.Count * TileSize;

            for (int x = halfX; x < maxX; x += (halfX * 2))
            {
                graphics.DrawLine(doorLinePen, new Point(x, 0), new Point(x, maxY));
            }

            for (int y = halfY; y < maxY; y += (halfY * 2))
            {
                graphics.DrawLine(doorLinePen, new Point(0,y), new Point(maxX, y));
            }

            if(this.SelectedEditMode == EditMode.DestructableBlock || this.SelectedEditMode == EditMode.ReadOnly)
            {
                //destructable blocks
                foreach (DestructableBlockPlacementDefinition block in worldMapCell.DestructableBlockPlacements)
                {
                    Rect rect = new Rect(block.X * TileSize, block.Y * TileSize, TileSize, TileSize);

                    DestructableBlockDefinition blockDefinition = this.session.RawContentManager.DestructableBlockRepository.GetContent(block.DestructableBlockKey);
                    TileSetDefinition tileSetDefinition = this.session.RawContentManager.TileSetRepository.GetContent(blockDefinition.TileSetKey);
                    IBitmap image = GetTileImage(tileSetDefinition);

                    graphics.DrawImage(image, rect);
                }
            }
            
            if (this.SelectedEditMode == EditMode.Door || this.SelectedEditMode == EditMode.ReadOnly)
            {
                //doors
                foreach(DoorPlacementDefinition door in worldMapCell.DoorPlacements)
                {
                    Rect rect = default;
                    IBitmap image = null;
                    if (door.DoorDirection == DoorDirection.Top || door.DoorDirection == DoorDirection.Bottom) rect = new Rect(door.X * TileSize, door.Y * TileSize, 4 * TileSize, TileSize);
                    else rect = new Rect(door.X * TileSize, door.Y * TileSize, TileSize, 4 * TileSize);

                    image = GetDoorCloseImage(door.DoorDirection);
                    graphics.DrawImage(image, rect);

                    image = GetDoorOpenImage(door.DoorDirection);
                    graphics.DrawImage(image, rect);
                }
            }

            if(this.SelectedEditMode == EditMode.Item || this.SelectedEditMode == EditMode.ReadOnly)
            {
                //items
                foreach(ItemPlacementDefinition item in worldMapCell.ItemPlacements)
                {
                    Rect rect = new Rect(item.X * TileSize, item.Y * TileSize, TileSize, TileSize);
                    IBitmap image = GetItemImage(item.ItemType);
                    graphics.DrawImage(image, rect);
                }
            }

            if(this.SelectedEditMode == EditMode.Enemy || this.SelectedEditMode == EditMode.ReadOnly)
            {
                //enemies
                foreach(EnemyPlacementDefinition enemy in worldMapCell.EnemyPlacements)
                {
                    Rect rect = new Rect(enemy.X * TileSize, enemy.Y * TileSize, TileSize, TileSize);
                    IBitmap image = GetEnemyImage(enemy.EnemyKey);
                    graphics.DrawImage(image, rect);
                }
            }

            if(this.SelectedEditMode == EditMode.Teleporter)
            {
                //teleporters
                foreach(TeleporterDefinition teleporter in worldMapCell.TeleporterPlacements)
                {
                    Rect rect = new Rect(teleporter.X * TileSize, teleporter.Y * TileSize, TileSize, TileSize);
                    graphics.FillRectangle(selectedPenBrush, rect);
                }
            }

            if(this.SelectedEditMode == EditMode.Saver)
            {
                foreach(SaverDefinition saver in worldMapCell.SaverPlacements)
                {
                    Rect rect = new Rect(saver.X * TileSize, saver.Y * TileSize, TileSize, TileSize);
                    graphics.FillRectangle(selectedPenBrush, rect);
                }
            }

            if (this.SelectedEditMode == EditMode.Victory)
            {
                foreach (VictoryDefinition victory in worldMapCell.VictoryPlacements)
                {
                    Rect rect = new Rect(victory.X * TileSize, victory.Y * TileSize, TileSize, TileSize);
                    graphics.FillRectangle(selectedPenBrush, rect);
                }
            }
        }


        private IBitmap GetItemImage(ItemType itemType)
        {
            if(cacheItemImages.ContainsKey(itemType) == false)
            {
                string file = @"textures\items\bomb-1";

                if(itemType == ItemType.Bombs) file = @"textures\items\bomb-1";
                else if (itemType == ItemType.Missiles) file = @"textures\items\missile-1";
                else if (itemType == ItemType.SuperMissiles) file = @"textures\items\supermissile-1";
                else if (itemType == ItemType.SuperBombs) file = @"textures\items\superbomb-1";
                else if (itemType == ItemType.MorphBall) file = @"textures\items\morphball-1";
                else if (itemType == ItemType.SpringBall) file = @"textures\items\springball-1";
                else if (itemType == ItemType.SuperJump) file = @"textures\items\superjump-1";
                else if (itemType == ItemType.VariaSuit) file = @"textures\items\varia-1";
                else if (itemType == ItemType.ChargeBeam) file = @"textures\items\chargebeam-1";
                else if (itemType == ItemType.WaveBeam) file = @"textures\items\wavebeam-1";
                else if (itemType == ItemType.PlasmaBeam) file = @"textures\items\plasmabeam-1";
                else if (itemType == ItemType.SpazerBeam) file = @"textures\items\spazerbeam-1";
                else if (itemType == ItemType.IceBeam) file = @"textures\items\icebeam-1";
                else if (itemType == ItemType.EnergyTank) file = @"textures\items\energytank-1";
                else if (itemType == ItemType.GravitySuit) file = @"textures\items\gravity-1";
                else if (itemType == ItemType.SpaceJump) file = @"textures\items\spacejump-1";
                else if (itemType == ItemType.ScrewAttack) file = @"textures\items\screwattack-1";
                else if (itemType == ItemType.SpeedBoost) file = @"textures\items\speedboost-1";

                string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + file.Replace('\\', Path.DirectorySeparatorChar) + ".png";
                cacheItemImages.Add(itemType, new Bitmap(filename));
            }

            return cacheItemImages[itemType];
        }


        private IBitmap GetEnemyImage(int enemyKey)
        {
            if(cacheEnemyImages.ContainsKey(enemyKey) == false)
            {
                EnemyDefinition enemy = this.session.RawContentManager.EnemyRepository.GetContent(enemyKey);
                RawResource texture = this.session.RawContentManager.TextureRepository.GetContent(enemy.TextureKey1);

                string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + texture.SystemFilePath + ".png";
                cacheEnemyImages.Add(enemyKey, new Bitmap(filename));
            }

            return cacheEnemyImages[enemyKey];
        }


        private IBitmap GetDoorCloseImage(DoorDirection doorDirection)
        {
            if(cacheDoorCloseImages.ContainsKey(doorDirection) == false)
            {
                string file = @"textures\doors\door-bottom-close";
                if(doorDirection == DoorDirection.Top) file = @"textures\doors\door-top-close";
                else if (doorDirection == DoorDirection.Left) file = @"textures\doors\door-left-close";
                else if (doorDirection == DoorDirection.Right) file = @"textures\doors\door-right-close";

                string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + file.Replace('\\',Path.DirectorySeparatorChar) + ".png";
                cacheDoorCloseImages.Add(doorDirection, new Bitmap(filename));
            }

            return cacheDoorCloseImages[doorDirection];
        }


        private IBitmap GetDoorOpenImage(DoorDirection doorDirection)
        {
            if (cacheDoorOpenImages.ContainsKey(doorDirection) == false)
            {
                string file = @"textures\doors\door-bottom-open";
                if (doorDirection == DoorDirection.Top) file = @"textures\doors\door-top-open";
                else if (doorDirection == DoorDirection.Left) file = @"textures\doors\door-left-open";
                else if (doorDirection == DoorDirection.Right) file = @"textures\doors\door-right-open";

                string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + file.Replace('\\', Path.DirectorySeparatorChar) + ".png";
                cacheDoorOpenImages.Add(doorDirection, new Bitmap(filename));
            }

            return cacheDoorOpenImages[doorDirection];
        }



        private IBitmap GetTileImage(TileSetDefinition tile)
        {            
            if (cacheTileImages.ContainsKey(tile.Key) == false)
            {
                string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + this.session.RawContentManager.TextureRepository.GetContent(tile.TextureKey).SystemFilePath + ".png";
                cacheTileImages.Add(tile.Key, ImageManip.GetImageSubset(filename, tile.X, tile.Y, tile.Width, tile.Height));
            }

            return cacheTileImages[tile.Key]; 
        }


        private async void WorldCellPreviewPanel_PointerPressed(object sender, Avalonia.Input.PointerPressedEventArgs e)
        {
            PointerPoint point = e.GetCurrentPoint(this);

#pragma warning disable CS0618 // Type or member is obsolete
            await ApplyMouse(point, e.ClickCount);
#pragma warning restore CS0618 // Type or member is obsolete
        }

        private async void WorldCellPreviewPanel_PointerMoved(object sender, Avalonia.Input.PointerEventArgs e)
        {
            PointerPoint point = e.GetCurrentPoint(this);

            await ApplyMouse(point, 1);
        }


        private async Task ApplyMouse(PointerPoint point, int clickCount)
        {
            if (this.worldMapCell == null) return;

            if (point.Properties.IsLeftButtonPressed || point.Properties.IsRightButtonPressed)
            {
                int tx = (int)point.Position.X / this.TileSize;
                int ty = (int)point.Position.Y / this.TileSize;

                if (this.SelectedEditMode == EditMode.Tile)
                {
                    ApplyTileSet(point, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.DestructableBlock)
                {
                    await ApplyDestructableBlock(clickCount, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.Door)
                {
                    await ApplyDoor(clickCount, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.Teleporter)
                {
                    await ApplyTeleporter(clickCount, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.Saver)
                {
                    ApplySaver(clickCount, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.Victory)
                {
                    ApplyVictory(clickCount, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.Item)
                {
                    await ApplyItem(clickCount, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.Enemy)
                {
                    await ApplyEnemy(clickCount, tx, ty);
                }
                else if (this.SelectedEditMode == EditMode.ReadOnly)
                {
                    this.SelectedTileX = tx;
                    this.SelectedTileY = ty;
                }
            }

            this.InvalidateVisual();
        }


        private async Task ApplyEnemy(int clickCount, int tx, int ty)
        {
            this.SelectedTileX = tx;
            this.SelectedTileY = ty;

            if (clickCount == 2)
            {
                EnemyPlacementDefinition placement = this.worldMapCell.GetEnemyPlacementAtPosition(tx, ty);

                int enemyKey = -1;
                if (placement != null) enemyKey = placement.EnemyKey;

                EnemySelectDialog dialog = new EnemySelectDialog();
                dialog.SetParameters(this.session, enemyKey);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == MessageBox.Avalonia.Enums.ButtonResult.Ok)
                {
                    if (placement == null)
                    {
                        placement = new EnemyPlacementDefinition()
                        {
                            X = tx,
                            Y = ty,
                            EnemyKey = dialog.SelectedEnemy.Key
                        };
                        this.worldMapCell.EnemyPlacements.Add(placement);
                    }
                    else
                    {
                        placement.EnemyKey = dialog.SelectedEnemy.Key;
                    }
                }
            }
        }



        private async Task ApplyItem(int clickCount, int tx, int ty)
        {
            this.SelectedTileX = tx;
            this.SelectedTileY = ty;

            if (clickCount == 2)
            {
                ItemPlacementDefinition placement = this.worldMapCell.GetItemPlacementAtPosition(tx, ty);
                ItemType itemType = ItemType.Missiles;

                if (placement != null)
                {
                    itemType = placement.ItemType;
                }

                ItemSelectDialog dialog = new ItemSelectDialog();
                dialog.SetParameters(itemType);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == MessageBox.Avalonia.Enums.ButtonResult.Ok)
                {
                    if (placement == null)
                    {
                        placement = new ItemPlacementDefinition()
                        {
                            WorldKey = worldMapCell.WorldKey,
                            CellX = worldMapCell.WorldMapX,
                            CellY = worldMapCell.WorldMapY,
                            X = tx,
                            Y = ty,
                            ItemType = dialog.SelectedItemType
                        };
                        this.worldMapCell.ItemPlacements.Add(placement);
                    }
                    else
                    {
                        placement.ItemType = dialog.SelectedItemType;
                    }
                }
            }
        }


        private void ApplyVictory(int clickCount, int tx, int ty)
        {
            this.SelectedTileX = tx;
            this.SelectedTileY = ty;

            if (clickCount == 2)
            {
                VictoryDefinition placement = this.worldMapCell.GetVictoryPlacementAtPosition(tx, ty);

                if (placement == null)
                {
                    placement = new VictoryDefinition()
                    {
                        X = tx,
                        Y = ty,
                    };
                    this.worldMapCell.VictoryPlacements.Add(placement);
                }
            }
        }


        private void ApplySaver(int clickCount, int tx, int ty)
        {
            this.SelectedTileX = tx;
            this.SelectedTileY = ty;

            if(clickCount == 2)
            {
                SaverDefinition placement = this.worldMapCell.GetSaverPlacementAtPosition(tx, ty);

                if(placement == null)
                {
                    placement = new SaverDefinition()
                    {
                        X = tx,
                        Y = ty,
                    };
                    this.worldMapCell.SaverPlacements.Add(placement);
                }
            }
        }


        private async Task ApplyTeleporter(int clickCount, int tx, int ty)
        {
            this.SelectedTileX = tx;
            this.SelectedTileY = ty;

            if(clickCount == 2)
            {
                TeleporterDefinition placement = this.worldMapCell.GetTeleporterPlacementAtPosition(tx, ty);

                WorldPosition destination = null;
                if (placement != null) destination = placement.Destination;

                WorldPositionSelectDialog dialog = new WorldPositionSelectDialog();
                dialog.SetSession(this.session);
                if(destination != null) dialog.SelectedWorldPosition = destination.Copy();

                await dialog.ShowDialog((Window)this.VisualRoot);

                if(dialog.DialogResult == MessageBox.Avalonia.Enums.ButtonResult.Ok)
                {
                    if(placement == null)
                    {
                        placement = new TeleporterDefinition()
                        {
                            X = tx,
                            Y = ty,
                            Destination = dialog.SelectedWorldPosition.Copy()
                        };
                        this.worldMapCell.TeleporterPlacements.Add(placement);
                    }
                    else
                    {
                        placement.Destination = dialog.SelectedWorldPosition.Copy();
                    }
                }
            }
        }


        private void ApplyTileSet(PointerPoint point, int tx, int ty)
        {
            if (this.TileSetSelectionPanel == null) return;

            TileSetDefinition tileToPaint = this.TileSetSelectionPanel.SelectedTileSet;

            if (tileToPaint == null) return;
            if (this.worldMapCell.TileSetKeysLayer1.ContainsKey(tx) == false) return;
            if (this.worldMapCell.TileSetKeysLayer1[tx].ContainsKey(ty) == false) return;

            if (this.IsControlDown)
            {
                if (point.Properties.IsLeftButtonPressed) this.worldMapCell.TileSetKeysLayer1[tx][ty] = null;
                else if (point.Properties.IsRightButtonPressed) this.worldMapCell.TileSetKeysLayer2[tx][ty] = null;
            }
            else if (this.IsShiftDown)
            {
                int maxX = this.worldMapCell.GetMaxTileX();
                int maxY = this.worldMapCell.GetMaxTileY();

                if (point.Properties.IsLeftButtonPressed)
                {
                    for (int x = 0; x < maxX; x++)
                    {
                        for (int y = 0; y < maxY; y++)
                        {
                            this.worldMapCell.TileSetKeysLayer1[x][y] = tileToPaint.Key;
                        }
                    }
                }
                else
                {
                    for (int x = 0; x < maxX; x++)
                    {
                        for (int y = 0; y < maxY; y++)
                        {
                            this.worldMapCell.TileSetKeysLayer2[x][y] = tileToPaint.Key;
                        }
                    }
                }
            }
            else
            {
                if (point.Properties.IsLeftButtonPressed) this.worldMapCell.TileSetKeysLayer1[tx][ty] = tileToPaint.Key;
                else this.worldMapCell.TileSetKeysLayer2[tx][ty] = tileToPaint.Key;
            }
        }
        

        private async Task ApplyDestructableBlock(int clickCount, int tx, int ty)
        {
            this.SelectedTileX = tx;
            this.SelectedTileY = ty;

            if (clickCount == 2)
            {
                DestructableBlockPlacementDefinition placement = this.worldMapCell.GetDestructableBlockPlacementAtPosition(tx, ty);
                int blockKey = -1;

                if (placement != null) blockKey = placement.DestructableBlockKey;

                DestructableBlockSelectDialog dialog = new DestructableBlockSelectDialog();
                dialog.SetParameters(this.session, blockKey);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == MessageBox.Avalonia.Enums.ButtonResult.Ok)
                {
                    DestructableBlockDefinition block = dialog.SelectedDestructableBlock;

                    if (placement == null)
                    {
                        placement = new DestructableBlockPlacementDefinition()
                        {
                            X = tx,
                            Y = ty,
                            DestructableBlockKey = block.Key
                        };

                        this.worldMapCell.DestructableBlockPlacements.Add(placement);
                    }
                    else
                    {
                        placement.DestructableBlockKey = block.Key;
                    }
                }

            }
        }


        private async Task ApplyDoor(int clickCount, int tx, int ty)
        {
            this.SelectedTileX = tx;
            this.SelectedTileY = ty;

            if (clickCount == 2)
            {
                DoorPlacementDefinition placement = this.worldMapCell.GetDoorPlacementAtPosition(tx, ty);
                
                DamageType damageType = DamageType.Normal;
                DoorDirection doorDirection = DoorDirection.Right;

                if(placement != null)
                {
                    damageType = placement.DamageType;
                    doorDirection = placement.DoorDirection;
                }

                DoorSelectDialog dialog = new DoorSelectDialog();
                dialog.SetParameters(damageType, doorDirection);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == MessageBox.Avalonia.Enums.ButtonResult.Ok)
                {
                    if(placement == null)
                    {
                        placement = new DoorPlacementDefinition()
                        {
                            X = tx,
                            Y = ty,
                            DamageType = dialog.SelectedDamageType,
                            DoorDirection = dialog.SelectedDoorDirection
                        };
                        this.worldMapCell.DoorPlacements.Add(placement);
                    }
                    else
                    {
                        placement.DamageType = dialog.SelectedDamageType;
                        placement.DoorDirection = dialog.SelectedDoorDirection;
                    }
                }
            }
        }

    }
}
