﻿using Avalonia;
using Avalonia.Controls;
using Avalonia.Input;
using Avalonia.Markup.Xaml;
using Avalonia.Media;
using GameMakerAvalonia.Forms;
using GameMakerAvalonia.Models;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameMakerAvalonia.PreviewPanels
{
    public class WorldPreviewPanel : UserControl
    {
        private Session session;
        private WorldMapDefinition worldMap;
        public bool AllowClicks { get; set; }
        public bool AllowDoubleClickToEdit { get; set; }
        public int CellSize { get; set; }


        public static readonly StyledProperty<WorldMapCellDefinition> SelectedCellProperty = AvaloniaProperty.Register<WorldPreviewPanel, WorldMapCellDefinition>(nameof(SelectedCell));
        public WorldMapCellDefinition SelectedCell
        {
            get { return GetValue(SelectedCellProperty); }
            set { SetValue(SelectedCellProperty, value); }
        }


        public WorldPreviewPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.SelectedCell = null;
            this.AllowClicks = true;
            this.AllowDoubleClickToEdit = false;
            this.CellSize = 32;

            this.PointerPressed += WorldPreviewPanel_PointerPressed;
        }


        public void SetSession(Session session)
        {
            this.session = session;
        }


        public void SetWorldMap(WorldMapDefinition worldMap)
        {
            this.worldMap = worldMap;
            this.SelectedCell = null;

            SetSize();
        }


        private void SetSize()
        {
            if(this.worldMap == null)
            {
                this.Height = 1;
                this.Width = 1;
                return;
            }

            IList<WorldMapCellDefinition> cells = this.worldMap.GetListOfCells();

            int maxX = 128 * this.CellSize;
            int maxY = 128 * this.CellSize;
            foreach (WorldMapCellDefinition cell in cells)
            {
                int x = (cell.WorldMapX * this.CellSize) + (cell.WorldMapWidth * this.CellSize);
                int y = (cell.WorldMapY * this.CellSize) + (cell.WorldMapHeight * this.CellSize);

                if (x > maxX) maxX = x;
                if (y > maxY) maxY = y;
            }

            this.Width = maxX;
            this.Height = maxY;


        }


        public override void Render(DrawingContext graphics)
        {
            base.Render(graphics);

            if ((this.session == null) || (this.worldMap == null)) return;

            IBrush backgroundBrush = new SolidColorBrush(Color.FromRgb(Convert.ToByte(0), Convert.ToByte(0), Convert.ToByte(0)));

            IBrush brush = new SolidColorBrush(Color.FromRgb(Convert.ToByte(160), Convert.ToByte(0), Convert.ToByte(160)));
            IBrush selectedBrush = new SolidColorBrush(Color.FromRgb(Convert.ToByte(192), Convert.ToByte(0), Convert.ToByte(192)));

            IBrush penBrush = new SolidColorBrush(Color.FromRgb(Convert.ToByte(224), Convert.ToByte(0), Convert.ToByte(224)));
            IPen pen = new Pen(penBrush);
            IBrush selectedPenBrush = new SolidColorBrush(Color.FromRgb(Convert.ToByte(255), Convert.ToByte(0), Convert.ToByte(0)));
            IPen selectedPen = new Pen(selectedPenBrush);

            graphics.FillRectangle(backgroundBrush, new Rect(0, 0, this.Width, this.Height));

            IList<WorldMapCellDefinition> cells = this.worldMap.GetListOfCells();
            foreach (WorldMapCellDefinition cell in cells)
            {
                int x = cell.WorldMapX * this.CellSize;
                int y = cell.WorldMapY * this.CellSize;
                int w = cell.WorldMapWidth * this.CellSize;
                int h = cell.WorldMapHeight * this.CellSize;

                Rect rect = new Rect(x, y, w, h);

                if (cell == this.SelectedCell) graphics.FillRectangle(selectedBrush, rect);
                else graphics.FillRectangle(brush, rect);

                graphics.DrawRectangle(pen, rect);
            }

            if (this.SelectedCell != null)
            {
                int x = this.SelectedCell.WorldMapX * this.CellSize;
                int y = this.SelectedCell.WorldMapY * this.CellSize;
                int w = this.SelectedCell.WorldMapWidth * this.CellSize;
                int h = this.SelectedCell.WorldMapHeight * this.CellSize;

                Rect rect = new Rect(x, y, w, h);

                graphics.DrawRectangle(selectedPen, rect);
            }
        }

        
        private void WorldPreviewPanel_PointerPressed(object sender, PointerPressedEventArgs e)
        {
#pragma warning disable CS0618 // Type or member is obsolete
            if (e.ClickCount == 2) OnDoubleClick(e);
#pragma warning restore CS0618 // Type or member is obsolete
            else OnClick(e);
        }


        private void OnClick(PointerPressedEventArgs e)
        {
            if (this.AllowClicks == false) return;
            if ((this.session == null) || (this.worldMap == null)) return;
            
            PointerPoint point = e.GetCurrentPoint(this);
            int tx = (int)point.Position.X;
            int ty = (int)point.Position.Y;

            int cx = tx / this.CellSize;
            int cy = ty / this.CellSize;

            this.SelectedCell = null;

            this.SelectedCell = worldMap.GetCellAtLocation(cx, cy);

            if (this.SelectedCell == null)
            {
                this.SelectedCell = new WorldMapCellDefinition()
                {
                    WorldKey = this.worldMap.Key,
                    WorldMapX = cx,
                    WorldMapY = cy,
                    WorldMapWidth = 1,
                    WorldMapHeight = 1
                };
            }

            this.InvalidateVisual();
        }


        private async void OnDoubleClick(PointerPressedEventArgs e)
        {
            if (this.AllowClicks == false) return;
            if ((this.session == null) || (this.worldMap == null)) return;

            PointerPoint point = e.GetCurrentPoint(this);
            int tx = (int)point.Position.X;
            int ty = (int)point.Position.Y;

            int cx = tx / this.CellSize;
            int cy = ty / this.CellSize;

            WorldMapCellDefinition cell = worldMap.GetCellAtLocation(cx, cy);

            if (cell == null) return;

            if(this.AllowDoubleClickToEdit)
            {
                WorldCellForm form = new WorldCellForm();
                form.SetSession(this.session);
                form.SetWorldMapCell(cell);
                await form.ShowDialog((Window)this.VisualRoot);
            }

        }


    }
}
