﻿using GameResources.Logic.Physics;
using GameResources.Manager;
using GameResources.Models.Effects;
using GameResources.Models.Samus;
using GameResources.Models.Weapons;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameResources.Models.Game
{
    public class GameState
    {
        private readonly IGameContentManager gameContentManager;
        public bool IsVictory { get; set; }
        public int GameSlot { get; set; }
        public Player Player { get; set; }
        public VariableState VariableState { get; set; }
        public MapHistory MapHistory { get; set; }
        public IList<ItemPlacementDefinition> PickedUpItems { get; set; }
        public IList<IWeaponSprite> ActiveWeaponSprites { get; private set; }
        public IList<IEffect> ActiveEffects { get; private set; }
        public IList<DestructableBlock> ActiveDestructableBlocks { get; private set; }
        public IList<Door> ActiveDoors { get; private set; }
        public IList<Teleporter> ActiveTeleporters { get; private set; }
        public IList<Item> ActiveItems { get; private set; }
        public IList<Saver> ActiveSavers { get; private set; }
        public IList<Enemy> ActiveEnemies { get; private set; }
        public IList<Victory> ActiveVictories { get; private set; }


        private WaterEffect waterEffect;

        private long lastHeatDamage;


        public GameState(IGameContentManager gameContentManager)
        {
            this.ActiveWeaponSprites = new List<IWeaponSprite>();
            this.ActiveEffects = new List<IEffect>();
            this.ActiveDestructableBlocks = new List<DestructableBlock>();
            this.ActiveDoors = new List<Door>();
            this.ActiveTeleporters = new List<Teleporter>();
            this.ActiveItems = new List<Item>();
            this.ActiveSavers = new List<Saver>();
            this.PickedUpItems = new List<ItemPlacementDefinition>();
            this.ActiveEnemies = new List<Enemy>();
            this.ActiveVictories = new List<Victory>();

            this.gameContentManager = gameContentManager;
            this.VariableState = new VariableState();
            this.MapHistory = new MapHistory();
            this.GameSlot = 1;
            this.IsVictory = false;

            this.lastHeatDamage = 0;
        }


        public void Update(long ticks)
        {
            //player
            this.Player.Update(this, gameContentManager, ticks);

            //weapon sprites
            for (int i = 0; i < ActiveWeaponSprites.Count; i++)
            {
                IWeaponSprite weaponSprite = ActiveWeaponSprites[i];

                weaponSprite.Update(this, gameContentManager, ticks);

                if (weaponSprite.IsExpired)
                {
                    ActiveWeaponSprites.Remove(weaponSprite);
                    i--;
                }
            }

            //effects
            for (int i = 0; i < ActiveEffects.Count; i++)
            {
                IEffect effect = ActiveEffects[i];

                effect.Update(ticks);

                if (effect.IsExpired)
                {
                    ActiveEffects.Remove(effect);
                    i--;
                }
            }

            //items 
            for(int i = 0; i < ActiveItems.Count; i++) ActiveItems[i].Update(this, gameContentManager, ticks);

            //destuctable blocks
            for (int i = 0; i < ActiveDestructableBlocks.Count; i++) ActiveDestructableBlocks[i].Update(this, gameContentManager, ticks);

            //doors
            for (int i = 0; i < ActiveDoors.Count; i++) ActiveDoors[i].Update(this, gameContentManager, ticks);

            //teleporters
            for (int i = 0; i < ActiveTeleporters.Count; i++) ActiveTeleporters[i].Update(this, gameContentManager, ticks);

            //savers
            for (int i = 0; i < ActiveSavers.Count; i++) ActiveSavers[i].Update(this, gameContentManager, ticks);

            //victories
            for (int i = 0; i < ActiveVictories.Count; i++) ActiveVictories[i].Update(this, gameContentManager, ticks);

            //enemies
            for (int i = 0; i < ActiveEnemies.Count; i++)
            {
                Enemy enemy = ActiveEnemies[i];

                enemy.Update(this, gameContentManager, ticks);

                if(enemy.IsExpired)
                {
                    ActiveEnemies.Remove(enemy);
                    i--;
                }
            }

            //tileset animations
            this.GetActiveWorldMapCell()?.UpdateTiles(ticks);

            //heated room
            lastHeatDamage += ticks;
            if(GetActiveWorldMapCell().IsHeated)
            {
                if (Player.Inventory.HasVariaSuit == false)
                {
                    if (lastHeatDamage >= (TimeSpan.TicksPerSecond * 0.1))
                    {
                        Player.CharacterStats.Health -= 1;
                        lastHeatDamage = 0;
                    }
                }
            }

            //item consumption
            Rectangle playerRect = this.Player.GetCurrentCollisionArea();
            for(int i = 0; i < this.ActiveItems.Count; i++)
            {
                Item item = this.ActiveItems[i];

                if(item.GetCurrentCollisionArea().Intersects(playerRect))
                {
                    this.Player.AddItem(item, gameContentManager, this);
                    this.ActiveItems.Remove(item);
                    i--;

                    if ((item.ItemPlacementDefinition != null) && (this.PickedUpItems.Contains(item.ItemPlacementDefinition) == false)) this.PickedUpItems.Add(item.ItemPlacementDefinition);
                }
            }
        }


        public void ActivateWorldMapCell()
        {
            WorldMapCell activeCell = this.GetActiveWorldMapCell();
            this.MapHistory.AddMapCell(activeCell);

            //clear
            ActiveWeaponSprites.Clear();
            ActiveEffects.Clear();
            ActiveDestructableBlocks.Clear();
            ActiveDoors.Clear();
            ActiveTeleporters.Clear();
            ActiveItems.Clear();
            ActiveSavers.Clear();
            ActiveEnemies.Clear();
            ActiveVictories.Clear();
            waterEffect = null;

            //create
            for(int i = 0; i < activeCell.DestructableBlockPlacements.Count; i++)
            {
                DestructableBlockPlacementDefinition placement = activeCell.DestructableBlockPlacements[i];

                DestructableBlock block = this.gameContentManager.DestructableBlockRepository.GetContentByKey(placement.DestructableBlockKey);
                block.Position.WorldMapKey = Player.Position.WorldMapKey;
                block.Position.WorldCellX = activeCell.WorldMapX;
                block.Position.WorldCellY = activeCell.WorldMapY;
                block.Position.PositionX = placement.X * TileSet.DRAW_SIZE;
                block.Position.PositionY = placement.Y * TileSet.DRAW_SIZE;

                ActiveDestructableBlocks.Add(block);
            }

            for(int i = 0; i < activeCell.DoorPlacements.Count; i++)
            {
                DoorPlacementDefinition placement = activeCell.DoorPlacements[i];

                Door door = new Door(gameContentManager, placement);
                ActiveDoors.Add(door);
            }

            for(int i = 0; i < activeCell.TeleporterPlacements.Count; i++)
            {
                TeleporterDefinition placement = activeCell.TeleporterPlacements[i];

                Teleporter teleporter = new Teleporter(placement, gameContentManager);
                ActiveTeleporters.Add(teleporter);
            }

            for (int i = 0; i < activeCell.ItemPlacements.Count; i++)
            {
                ItemPlacementDefinition placement = activeCell.ItemPlacements[i];

                if (this.PickedUpItems.Contains(placement) == false)
                {
                    Item item = new Item(placement, gameContentManager);
                    ActiveItems.Add(item);
                }
            }

            for (int i = 0; i < activeCell.SaverPlacements.Count; i++)
            {
                SaverDefinition placement = activeCell.SaverPlacements[i];

                Saver saver = new Saver(placement, gameContentManager);
                ActiveSavers.Add(saver);
            }

            for (int i = 0; i < activeCell.VictoryPlacements.Count; i++)
            {
                VictoryDefinition placement = activeCell.VictoryPlacements[i];

                Victory victory = new Victory(placement, gameContentManager);
                ActiveVictories.Add(victory);
            }

            for (int i = 0; i < activeCell.EnemyPlacements.Count; i++)
            {
                EnemyPlacementDefinition placement = activeCell.EnemyPlacements[i];

                Enemy enemy = gameContentManager.EnemyRepository.GetContentByKey(placement.EnemyKey);
                enemy.Position.WorldMapKey = activeCell.WorldKey;
                enemy.Position.WorldCellX = activeCell.WorldMapX;
                enemy.Position.WorldCellY = activeCell.WorldMapY;
                enemy.Position.PositionX = placement.X * TileSet.DRAW_SIZE;
                enemy.Position.PositionY = placement.Y * TileSet.DRAW_SIZE;
                enemy.Position.PositionX += (TileSet.DRAW_SIZE / 2);
                enemy.Position.PositionY += (TileSet.DRAW_SIZE);

                EnemyPositionAdjuster adjuster = new EnemyPositionAdjuster(enemy, this);
                adjuster.AdjustIfNeeded();

                if (enemy.OnlyShowIfVariableIsFalse.Equals(""))
                {
                    ActiveEnemies.Add(enemy);
                }
                else
                {
                    bool variableState = this.VariableState.GetVariable(enemy.OnlyShowIfVariableIsFalse);

                    if(variableState == false)
                    {
                        ActiveEnemies.Add(enemy);
                    }
                }
            }

            if(activeCell.IsHeated)
            {
                HeatedRoomEffect heatedRoomEffect = new HeatedRoomEffect(gameContentManager.TextureRepository, activeCell.GetMaxTileX() * TileSet.DRAW_SIZE, activeCell.GetMaxTileY() * TileSet.DRAW_SIZE);
                ActiveEffects.Add(heatedRoomEffect);
            }

            if(activeCell.WaterLevel > 0)
            {
                WaterEffect water = new WaterEffect(gameContentManager.TextureRepository, activeCell, activeCell.WaterLevel);
                ActiveEffects.Add(water);

                waterEffect = water;
            }
        }


        public bool IsBossInRoom()
        {
            for(int i = 0; i < this.ActiveEnemies.Count; i++) 
            {
                if (this.ActiveEnemies[i].IsBoss) return true;
            }

            return false;
        }


        public bool IsInteractingWithWater()
        {
            if (waterEffect == null) return false;

            Rectangle playerBounds = Player.GetCurrentCollisionArea();
            Rectangle waterBounds = this.waterEffect.GetCurrentCollisionArea();

            return (playerBounds.Intersects(waterBounds));
        }


        public Saver IsInteractingWithASaver()
        {
            for(int i = 0; i < ActiveSavers.Count; i++)
            {
                Saver saver = ActiveSavers[i];

                if (saver.IsPlayerOn) return saver;
            }

            return null;
        }


        public Victory IsInteractingWithAVictory()
        {
            for(int i = 0; i < ActiveVictories.Count; i++)
            {
                Victory victory = ActiveVictories[i];

                if (victory.IsPlayerOn) return victory;
            }

            return null;
        }


        public Teleporter IsInteractingWithATeleporter()
        {
            for(int i = 0; i < ActiveTeleporters.Count; i++)
            {
                Teleporter teleporter = ActiveTeleporters[i];

                if (teleporter.IsPlayerOn) return teleporter;
            }

            return null;
        }


        public WorldMapCell GetActiveWorldMapCell()
        {
            if (this.Player == null) return null;

            return GetWorldMapCell(this.Player.Position);
        }


        public WorldMapCell GetWorldMapCell(WorldPosition position)
        {
            var worldMapRepository = this.gameContentManager.WorldMapRepository;

            if (worldMapRepository.GetContentByKey(position.WorldMapKey) != null)
            {
                if (worldMapRepository.GetContentByKey(position.WorldMapKey).Cells.ContainsKey(position.WorldCellX))
                {
                    if (worldMapRepository.GetContentByKey(position.WorldMapKey).Cells[position.WorldCellX].ContainsKey(position.WorldCellY))
                    {
                        return this.gameContentManager.WorldMapRepository.GetContentByKey(position.WorldMapKey).Cells[position.WorldCellX][position.WorldCellY];
                    }
                }
            }

            return null;
        }


        public WorldMap GetActiveWorldMap()
        {
            if (this.Player == null) return null;

            return this.gameContentManager.WorldMapRepository.GetContentByKey(this.Player.Position.WorldMapKey);
        }


        public override string ToString()
        {
            string text = this.GameSlot + " - " + this.Player.CharacterStats.MaximumHealth;

            return text;
        }


    }
}
