﻿using GameResources.Logic.Physics;
using GameResources.Logic.Weapons;
using GameResources.Manager;
using GameResources.Models.Characters;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameResources.Models.Weapons
{
    public class BasicShot : IWeaponSprite
    {
        public WorldPosition Position { get; set; }
        public Motion Motion { get; set; }
        public bool IsExpired { get; set; }
        public DamageType DamageType { get; private set; }
        public WeaponModifier WeaponModifier { get; set; }
        public int Damage { get; private set; }
        
        private readonly int size;
        private readonly IGameContentManager gameContentManager;
        private readonly GameState gameState;
        private int activeTexture;
        private readonly IList<Texture2D> textures;
        private readonly IBattleObject caster;
        
        private long lastWaveUpdate;
        private long lastAnimationUpdate;
        private long lastMoveUpdate;
        private long creationTime;
        private IList<Enemy> hitEnemies;

        private int waveDeltaX;
        private int waveDeltaY;


        public BasicShot(GameState gameState, IGameContentManager gameContentManager, IBattleObject caster, WeaponModifier weaponModifier)
        {
            this.gameContentManager = gameContentManager;
            this.gameState = gameState;
            this.WeaponModifier = weaponModifier;
            this.caster = caster;

            this.activeTexture = 0;
            this.textures = new List<Texture2D>();
            this.hitEnemies = new List<Enemy>();

            if (weaponModifier.IsPlasma)
            {
                this.textures.Add(gameContentManager.TextureRepository.GetContentByName(@"textures\sprites\beam-1"));
                this.textures.Add(gameContentManager.TextureRepository.GetContentByName(@"textures\sprites\beam-2"));
                this.size = weaponModifier.IsCharged ? 24 : 16;
                this.Damage = 50;
            }
            else if (weaponModifier.IsSpazer)
            {
                this.textures.Add(gameContentManager.TextureRepository.GetContentByName(@"textures\sprites\ice-beam-1"));
                this.textures.Add(gameContentManager.TextureRepository.GetContentByName(@"textures\sprites\ice-beam-2"));
                this.size = weaponModifier.IsCharged ? 6 : 4;
                this.Damage = 30;
            }
            else
            {
                this.textures.Add(gameContentManager.TextureRepository.GetContentByName(@"textures\sprites\basic"));
                this.size = weaponModifier.IsCharged ? 32 : 16;
                this.Damage = 10;
            }

            this.IsExpired = false;
            this.Motion = new Motion();
            this.Position = new WorldPosition();

            this.waveDeltaX = 0;
            this.waveDeltaY = 0;

            if(weaponModifier.IsCharged)
            {
                this.DamageType = DamageType.Charged;
                this.Damage = this.Damage * 3;
            }
            else
            {
                this.DamageType = DamageType.Normal;
            }

            this.creationTime = 0;
            this.lastMoveUpdate = 0;
            this.lastAnimationUpdate = 0;
            this.lastWaveUpdate = 0;
        }


        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            Texture2D texture = this.textures[activeTexture];

            Vector2 position = new Vector2((int)(Position.PositionX - dx + waveDeltaX), (int)(Position.PositionY - dy + waveDeltaY));
            float scale = size / texture.Height;
            Rectangle sourceRect = new Rectangle(0, 0, texture.Width, texture.Height);
            
            float angle = (float)Math.Atan2(Motion.VelocityY, Motion.VelocityX) + (float)(Math.PI);

            Color color = Color.White;
            if(WeaponModifier.IsIce)
            {
                color = Color.CornflowerBlue;
            }
            else if (WeaponModifier.IsPlasma)
            {
                color = Color.LightGreen;
            }
            else if (WeaponModifier.IsSpazer)
            {
                color = Color.Yellow;
            }

            spriteBatch.Draw(texture, position, sourceRect, color, angle, new Vector2(texture.Width/2, texture.Height/2), scale, SpriteEffects.None, 0);
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            creationTime += ticks;
            lastWaveUpdate += ticks;
            lastMoveUpdate += ticks;
            lastAnimationUpdate += ticks;

            WorldMapCell worldMapCell = gameState.GetActiveWorldMapCell();

            //animate
            if (lastAnimationUpdate >= (TimeSpan.TicksPerSecond * 0.1))
            {
                UpdateAnimation();
                lastAnimationUpdate = 0;
            }

            //wave
            if (WeaponModifier.IsWave)
            {
                if (lastWaveUpdate >= (TimeSpan.TicksPerSecond * 0.025))
                {
                    UpdateWave();
                    lastWaveUpdate = 0;
                }
            }

            //move object
            if (lastMoveUpdate >= (TimeSpan.TicksPerSecond * 0.01))
            {
                PhysicsEngine physics = new PhysicsEngine(this, worldMapCell, gameState, true)
                {
                    IgnoreAllCollisions = true
                };
                physics.IgnoreObjects.Add(caster);
                physics.Update();

                lastMoveUpdate = 0;
            }

            //detect hit and damage enemy
            if (DetectCollisionWithDamagables() && (WeaponModifier.IsPlasma == false))
            {
                IsExpired = true;
            }
            else if (HasCollidedWithMap(worldMapCell) && (WeaponModifier.IsWave == false))
            {
                IsExpired = true;
            }
            else if (creationTime >= (TimeSpan.TicksPerSecond * 5))
            {
                IsExpired = true;
            }
        }


        private void UpdateAnimation()
        {
            activeTexture++;
            if (activeTexture >= this.textures.Count) activeTexture = 0;
        }


        private void UpdateWave()
        {
            double t = (double)creationTime / TimeSpan.TicksPerSecond;

            if (this.Motion.VelocityX != 0) waveDeltaY = (int)(Math.Sin(t * 180) * 16);
            if (this.Motion.VelocityY != 0) waveDeltaX = (int)(Math.Sin(t * 180) * 16);
        }


        private bool HasCollidedWithMap(WorldMapCell worldMapCell)
        {
            CollisionDetection collisionDetection = new CollisionDetection(this, worldMapCell, this.gameState);

            return collisionDetection.HasCollidedInWorld();
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(Position.PositionX + waveDeltaX, Position.PositionY + waveDeltaY);
        }


        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            Rectangle area = new Rectangle((int)(x - 1), (int)(y - 1), 3, 3);

            return area;
        }


        private bool DetectCollisionWithDamagables()
        {
            WeaponCollisionDetection weaponCollisionDetection = new WeaponCollisionDetection(gameState, gameContentManager, this, hitEnemies);
            return weaponCollisionDetection.DealWithCollision();
        } 


    }
}
