﻿using GameResources.Logic.Physics;
using GameResources.Logic.Weapons;
using GameResources.Manager;
using GameResources.Models.Characters;
using GameResources.Models.Effects;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System;


namespace GameResources.Models.Weapons
{
    public class MissileShot : IWeaponSprite
    {
        public WorldPosition Position { get; set; }
        public Motion Motion { get; set; }
        public bool IsExpired { get; set; }
        public DamageType DamageType { get; private set; }
        public WeaponModifier WeaponModifier { get; set; }
        public int Damage { get; private set; }

        private readonly IGameContentManager gameContentManager;
        private readonly int size;
        private readonly GameState gameState;
        private readonly Texture2D texture;
        private readonly IBattleObject caster;

        private long lastSpeedUpdate;
        private long lastMoveUpdate;
        private long creationTime;


        public MissileShot(GameState gameState, IGameContentManager gameContentManager, IBattleObject caster)
        {
            this.gameContentManager = gameContentManager;
            this.gameState = gameState;
            this.texture = gameContentManager.TextureRepository.GetContentByName(@"textures\sprites\missile");
            this.size = 32;
            this.caster = caster;

            this.IsExpired = false;
            this.Motion = new Motion();
            this.Position = new WorldPosition();
            this.DamageType = DamageType.Missile;
            this.Damage = 100;
            this.WeaponModifier = new WeaponModifier();

            this.lastMoveUpdate = 0;
            this.lastSpeedUpdate = 0;
            this.creationTime = 0;
        }


        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            Vector2 position = new Vector2((int)(Position.PositionX - dx), (int)(Position.PositionY - dy));
            float scale = size / this.texture.Width;
            Rectangle sourceRect = new Rectangle(0, 0, this.texture.Width, this.texture.Height);

            float angle = (float)Math.Atan2(Motion.VelocityY, Motion.VelocityX) + (float)(Math.PI);
            spriteBatch.Draw(this.texture, position, sourceRect, Color.White, angle, new Vector2(this.texture.Width / 2, this.texture.Height / 2), scale, SpriteEffects.None, 0);
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            lastMoveUpdate += ticks;
            lastSpeedUpdate += ticks;
            creationTime += ticks;

            WorldMapCell worldMapCell = gameState.GetActiveWorldMapCell();

            //accelerate object
            if (lastSpeedUpdate >= (TimeSpan.TicksPerSecond * 0.1))
            {
                if (this.Motion.VelocityX > 0) this.Motion.VelocityX += 1.5f;
                if (this.Motion.VelocityY > 0) this.Motion.VelocityY += 1.5f;

                if (this.Motion.VelocityX < 0) this.Motion.VelocityX += -1.5f;
                if (this.Motion.VelocityY < 0) this.Motion.VelocityY += -1.5f;

                lastSpeedUpdate = 0;
            }

            //move object
            if (lastMoveUpdate >= (TimeSpan.TicksPerSecond * 0.01))
            {
                PhysicsEngine physics = new PhysicsEngine(this, worldMapCell, gameState, true)
                {
                    IgnoreAllCollisions = true
                };
                physics.IgnoreObjects.Add(caster);
                physics.Update();

                lastMoveUpdate = 0;
            }

            //detect hit and damage enemy
            if (DetectCollisionWithDamagables())
            {
                IsExpired = true;

                MissileExplosionEffect effect = new MissileExplosionEffect(gameContentManager.TextureRepository, gameContentManager.SoundEffectRepository, this.Position.PositionX, this.Position.PositionY);
                gameState.ActiveEffects.Add(effect);
            }
            else if (HasCollidedWithMap(worldMapCell))
            {
                IsExpired = true;

                MissileExplosionEffect effect = new MissileExplosionEffect(gameContentManager.TextureRepository, gameContentManager.SoundEffectRepository, this.Position.PositionX, this.Position.PositionY);
                gameState.ActiveEffects.Add(effect);
            }
            else if (creationTime >= (TimeSpan.TicksPerSecond * 5))
            {
                IsExpired = true;
            }
        }


        private bool HasCollidedWithMap(WorldMapCell worldMapCell)
        {
            CollisionDetection collisionDetection = new CollisionDetection(this, worldMapCell, gameState);

            return collisionDetection.HasCollidedInWorld();
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(Position.PositionX, Position.PositionY);
        }


        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            Rectangle area = new Rectangle((int)(x - 1), (int)(y - 1), 3, 3);

            return area;
        }


        private bool DetectCollisionWithDamagables()
        {
            WeaponCollisionDetection weaponCollisionDetection = new WeaponCollisionDetection(gameState, gameContentManager, this, null);
            return weaponCollisionDetection.DealWithCollision();
        }


    }
}
