﻿using GameResources.Manager;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;


namespace GameResources.Models.World
{
    public class Door : IMoveableObject
    {
        public WorldPosition Position { get ; set; }
        public Motion Motion { get; set; }
        public DamageType DamageType { get; set; }
        public DoorDirection DoorDirection { get; set; }

        private readonly IGameContentManager gameContentManager;
        private Texture2D openTexture;
        private Texture2D closeTexture;
        private Color closeMask;
        public bool IsDestroyed { get; set; }


        public Door(IGameContentManager gameContentManager, DoorPlacementDefinition definition)
        {
            this.gameContentManager = gameContentManager;

            this.Motion = new Motion();
            this.Position = new WorldPosition()
            {
                PositionX = definition.X * TileSet.DRAW_SIZE,
                PositionY = definition.Y * TileSet.DRAW_SIZE
            };
            this.IsDestroyed = false;
            this.DamageType = definition.DamageType;
            this.DoorDirection = definition.DoorDirection;

            if (this.DoorDirection == DoorDirection.Top) openTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-top-open");
            else if (this.DoorDirection == DoorDirection.Bottom) openTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-bottom-open");
            else if (this.DoorDirection == DoorDirection.Left) openTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-left-open");
            else if (this.DoorDirection == DoorDirection.Right) openTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-right-open");

            if (this.DoorDirection == DoorDirection.Top) closeTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-top-close");
            else if (this.DoorDirection == DoorDirection.Bottom) closeTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-bottom-close");
            else if (this.DoorDirection == DoorDirection.Left) closeTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-left-close");
            else if (this.DoorDirection == DoorDirection.Right) closeTexture = gameContentManager.TextureRepository.GetContentByName(@"textures\doors\door-right-close");

            if (DamageType == DamageType.Missile) closeMask = new Color(248,96,96);
            else if (DamageType == DamageType.SuperMissile) closeMask = Color.LightGreen;
            else if (DamageType == DamageType.SuperBomb) closeMask = Color.Yellow;
            else closeMask = new Color(56,232,248);
        }



        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            Rectangle rect = (this.DoorDirection == DoorDirection.Top || this.DoorDirection == DoorDirection.Bottom) ?
                                new Rectangle((int)(this.Position.PositionX - dx), (int)(this.Position.PositionY - dy), 4 * TileSet.DRAW_SIZE, TileSet.DRAW_SIZE) :
                                new Rectangle((int)(this.Position.PositionX - dx), (int)(this.Position.PositionY - dy), TileSet.DRAW_SIZE, 4 * TileSet.DRAW_SIZE);

            spriteBatch.Draw(openTexture, rect, Color.White);
            if (IsDestroyed == false) spriteBatch.Draw(closeTexture, rect, closeMask);
        }


        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            int w = 0;
            int h = 0;

            if (this.DoorDirection == DoorDirection.Top || this.DoorDirection == DoorDirection.Bottom)
            {
                w = TileSet.DRAW_SIZE * 4;
                h = TileSet.DRAW_SIZE;
            }
            else
            {
                w = TileSet.DRAW_SIZE;
                h = TileSet.DRAW_SIZE * 4;
            }

            return new Rectangle((int)x, (int)y, w, h);
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(this.Position.PositionX, this.Position.PositionY);
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            
        }


        public void AttemptDestruction(DamageType damageType, GameState gameState, IGameContentManager gameContentManager)
        {
            if (this.IsDestroyed) return;

            //destruction test
            if (this.DamageType == DamageType.Missile)
            {
                if (damageType == DamageType.Missile || damageType == DamageType.SuperMissile)
                {
                    this.IsDestroyed = true;
                }
            }
            else if (this.DamageType == DamageType.Bomb)
            {
                if (damageType == DamageType.Bomb || damageType == DamageType.SuperBomb)
                {
                    this.IsDestroyed = true;
                }
            }
            else if (this.DamageType == DamageType.Normal)
            {
                this.IsDestroyed = true;
            }
            else if (this.DamageType == damageType)
            {
                this.IsDestroyed = true;
            }

            if (this.IsDestroyed)
            {
                //play door open sound
                this.gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Door Open").Play();
            }
        }


    }
}
