﻿using GameResources.Models.World;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using RawResources.Repository;
using System.Collections.Generic;


namespace GameResources.Repository
{
    public class CachedTileSetRepository : IGameContentRepository<TileSet>
    {
        private readonly IRawContentRepository<TileSetDefinition> definitionRepository;
        private readonly IGameNamedContentRepository<Texture2D> textureRepository;
        private readonly IDictionary<int, TileSet> cache;


        public CachedTileSetRepository(IRawContentRepository<TileSetDefinition> definitionRepository, IGameNamedContentRepository<Texture2D> textureRepository)
        {
            this.definitionRepository = definitionRepository;
            this.textureRepository = textureRepository;
            this.cache = new Dictionary<int, TileSet>();
        }


        public TileSet GetContentByKey(int key)
        {
            if (this.cache.ContainsKey(key)) return this.cache[key];

            TileSetDefinition definition = this.definitionRepository.GetContent(key);
            if (definition == null) return null;

            Load(definition);

            return this.cache[key];
        }


        private void Load(TileSetDefinition definition)
        {
            Texture2D rawTexture = this.textureRepository.GetContentByKey(definition.TextureKey);

            Texture2D tileTexture = new Texture2D(rawTexture.GraphicsDevice, definition.Width, definition.Height);
            Rectangle tileRect = new Rectangle(definition.X, definition.Y, definition.Width, definition.Height);
            Color[] tileColor = new Color[definition.Width * definition.Height];

            rawTexture.GetData(0, tileRect, tileColor, 0, tileColor.Length);
            tileTexture.SetData(tileColor);

            TileSet tileSet = new TileSet()
            {
                Key = definition.Key,
                Texture = tileTexture,
                Passable = definition.Passable
            };

            foreach (int i in definition.AnimationSequence)
            {
                if (i == definition.Key)
                {
                    tileSet.AnimationSequence.Add(tileSet);
                }
                else
                {
                    tileSet.AnimationSequence.Add(this.GetContentByKey(i));
                }
            }

            this.cache.Add(tileSet.Key, tileSet);
        }

    }
}
