﻿using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;


namespace RawResources.Repository
{
    public class FileTileSetRepository : IRawContentRepository<TileSetDefinition>
    {
        private IDictionary<int, TileSetDefinition> contentsByKey;
        private IList<int> keys;


        public FileTileSetRepository(string filename)
        {
            Load(filename);
        }


        public TileSetDefinition GetContent(int key)
        {
            TileSetDefinition content = null;

            if (this.contentsByKey.ContainsKey(key))
            {
                content = this.contentsByKey[key];
            }

            return content;
        }


        public IList<TileSetDefinition> GetListOfContent()
        {
            IList<TileSetDefinition> content = new List<TileSetDefinition>();
            IList<int> keys = GetKeys();

            foreach (int key in keys)
            {
                content.Add(contentsByKey[key]);
            }

            return content;
        }


        public void AddContent(TileSetDefinition content)
        {
            int key = GetMaxKey() + 1;

            content.Key = key;

            this.contentsByKey.Add(key, content);
            this.keys.Add(key);
        }


        public void RemoveContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                this.contentsByKey.Remove(key);
                this.keys.Remove(key);
            }
        }


        private int GetMaxKey()
        {
            int maxKey = -1;

            foreach (int key in this.keys)
            {
                if (maxKey < key) maxKey = key;
            }

            return maxKey;
        }


        public IList<int> GetKeys()
        {
            return this.keys;
        }


        public void Load(string filename)
        {
            Stream stream = null;
            StreamReader reader = null;
            string line = null;
            string[] lineContents = null;

            this.contentsByKey = new Dictionary<int, TileSetDefinition>();
            this.keys = new List<int>();

            try
            {
                stream = File.OpenRead(filename);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return;
            }

            //skip first line
            reader.ReadLine();

            while ((line = reader.ReadLine()) != null)
            {
                lineContents = line.Split(',');

                if (lineContents[0].Equals("") == false)
                {
                    LoadContent(lineContents);
                }
            }

            reader.Close();
        }


        private void LoadContent(string[] lineContents)
        {
            TileSetDefinition definition = new TileSetDefinition();

            definition.Key = Int32.Parse(lineContents[0]);
            definition.TextureKey = Int32.Parse(lineContents[1]);
            definition.X = Int32.Parse(lineContents[2]);
            definition.Y = Int32.Parse(lineContents[3]);
            definition.Width = Int32.Parse(lineContents[4]);
            definition.Height = Int32.Parse(lineContents[5]);
            definition.Passable = Boolean.Parse(lineContents[6]);

            if (lineContents[7].Equals("") == false)
            {
                definition.AnimationSequence = new List<int>(Array.ConvertAll(lineContents[7].Split('|'), Int32.Parse));
            }
            else
            {
                definition.AnimationSequence = new List<int>();
            }

            this.contentsByKey.Add(definition.Key, definition);
            this.keys.Add(definition.Key);
        }


        public void Save(string filename)
        {
            StreamWriter writer = new StreamWriter(filename, false);

            writer.WriteLine(TileSetDefinition.GetCsvTitle());

            foreach(int key in this.keys)
            {
                TileSetDefinition tileSet = this.contentsByKey[key];

                writer.WriteLine(tileSet.ToCsvString());
            }
            
            writer.Close();
        }


    }
}
