﻿using GameResources.Manager;
using GameResources.Models.Game;
using GameResources.Models.Samus;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using System;
using System.Text;
using UserInterface.Control;
using UserInterface.Core;


namespace UserInterface.Screen
{
    public class GameMenuScreen : IScreen
    {
        public ControlCollection Controls { get; }
        private readonly IScreenController screenController;

        private long lastMovementInput;
        private long lastActionInput;
        private MiniMap miniMap;
        private Label inventoryLabel;


        public GameMenuScreen(IScreenController screenController)
        {
            this.Controls = new ControlCollection();
            this.screenController = screenController;

            Init();
        }


        public void Draw(SpriteBatch spriteBatch)
        {
            for (int i = 0; i < this.Controls.Count; i++)
            {
                IControl control = this.Controls[i];
                control.Draw(spriteBatch);
            }
        }


        public void Update(long ticks)
        {
            InputMap inputMap = screenController.GetInputMap();

            lastActionInput += ticks;
            lastMovementInput += ticks;

            //keyboard
            if (lastMovementInput >= (TimeSpan.TicksPerSecond * 0.12))
            {
                if (inputMap.Up)
                {
                    this.Controls.SelectPrev();
                    lastMovementInput = 0;
                }
                else if (inputMap.Down)
                {
                    this.Controls.SelectNext();
                    lastMovementInput = 0;
                }
            }

            if (lastActionInput >= (TimeSpan.TicksPerSecond * 0.3))
            {
                if (inputMap.Jump)
                {
                    if (this.Controls.IndexOfSelected() != -1)
                    {
                        lastActionInput = 0;
                        screenController.LastKeyboardInputTicks = 0;

                        this.Controls[this.Controls.IndexOfSelected()].Press();
                    }
                }
                else if (inputMap.ChangeItem)
                {
                    lastActionInput = 0;
                    screenController.LastKeyboardInputTicks = 0;

                    this.screenController.ChangeScreen(ScreenType.Game);
                }
            }

            //controls
            foreach (IControl control in this.Controls)
            {
                control.Update(ticks);
            }
        }


        public void ActivateScreen()
        {
            lastMovementInput = screenController.LastKeyboardInputTicks;
            lastActionInput = screenController.LastKeyboardInputTicks;

            miniMap.GameState = screenController.GameState;
            inventoryLabel.Text = GetInventoryText();
        }


        private void Init()
        {
            IGameContentManager gameContentManager = this.screenController.GameContentManager;

            int mapCellSize = 16;
            int mapCells = 35;
            int mapWidth = mapCellSize * mapCells;
            miniMap = new MiniMap(gameContentManager.TextureRepository, gameContentManager.FontRepository, mapCells, new Color(50, 50, 50))
            {
                Location = new Rectangle(Resolution.RESOLUTION_X - mapWidth - 10, 10, mapWidth, mapWidth),
            };
            Controls.Add(miniMap);

            int buttonWidth = 220;
            int buttonHeight = 32;
            int buttonStartX = 120;
            int buttonStartY = 20;
            int buttonSpace = 48;

            this.Controls.Add(new Button()
            {
                Text = "Back to Game",
                Location = new Rectangle(buttonStartX, buttonStartY, buttonWidth, buttonHeight),
                TextureNormal = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button"),
                TextureSelected = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-select"),
                TexturePressed = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-press"),
                Font = gameContentManager.FontRepository.GetContentByName(@"fonts\Normal"),
                PressSound = gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Samus Land"),
                IsSelected = true,
                CommandText = "2"
            });
            ((Button)this.Controls[this.Controls.Count - 1]).OnClick += BackToGame;

            this.Controls.Add(new Button()
            {
                Text = "Main Menu",
                Location = new Rectangle(buttonStartX, buttonStartY + (buttonSpace * 1), buttonWidth, buttonHeight),
                TextureNormal = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button"),
                TextureSelected = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-select"),
                TexturePressed = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-press"),
                Font = gameContentManager.FontRepository.GetContentByName(@"fonts\Normal"),
                PressSound = gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Samus Land"),
                IsSelected = false,
                CommandText = "2"
            });
            ((Button)this.Controls[this.Controls.Count - 1]).OnClick += MainMenu;

            inventoryLabel = new Label()
            {
                Font = gameContentManager.FontRepository.GetContentByName(@"fonts\Normal"),
                Text = GetInventoryText(),
                Location = new Rectangle(10, 120, 600, 400)
            };
            this.Controls.Add(inventoryLabel);
        }


        private string GetInventoryText()
        {
            Player player = screenController.GameState?.Player;

            if (player == null) return "";

            StringBuilder builder = new StringBuilder();

            builder.AppendLine("Weapons:");
            if (player.Inventory.HasChargeBeam) builder.AppendLine("-Charge Beam");
            if (player.Inventory.HasWaveBeam) builder.AppendLine("-Wave Beam");
            if (player.Inventory.HasSpazerBeam) builder.AppendLine("-Spazer Beam");
            if (player.Inventory.HasIceBeam) builder.AppendLine("-Ice Beam");
            if (player.Inventory.HasPlasmaBeam) builder.AppendLine("-Plasma Beam");

            builder.AppendLine("");

            builder.AppendLine("Consumeables:");
            builder.AppendLine("-Missiles       " + player.Inventory.Missiles + "/" + player.Inventory.MissilesMax);
            builder.AppendLine("-Super Missiles " + player.Inventory.SuperMissiles + "/" + player.Inventory.SuperMissilesMax);
            builder.AppendLine("-Super Bombs    " + player.Inventory.SuperBombs + "/" + player.Inventory.SuperBombsMax);

            builder.AppendLine("");

            builder.AppendLine("Suits:");
            if (player.Inventory.HasVariaSuit) builder.AppendLine("-Varia");
            if (player.Inventory.HasGravitySuit) builder.AppendLine("-Gravity");

            builder.AppendLine("");

            builder.AppendLine("Abilities:");
            if (player.Inventory.HasMorphBall) builder.AppendLine("-Morph Ball");
            if (player.Inventory.HasBombs) builder.AppendLine("-Bombs");
            if (player.Inventory.HasSpringBall) builder.AppendLine("-Spring Ball");

            if (player.Inventory.HasSuperJump) builder.AppendLine("-High Jump");
            if (player.Inventory.HasSpeedBoost) builder.AppendLine("-Speed Boost");
            if (player.Inventory.HasSpaceJump) builder.AppendLine("-Space Jump");
            if (player.Inventory.HasScrewAttack) builder.AppendLine("-Screw Attack");

            return builder.ToString();
        }


        private void MainMenu(object sender)
        {
            this.screenController.GameState = null;

            this.screenController.ChangeScreen(ScreenType.MainMenu);
        }


        private void BackToGame(object sender)
        {
            this.screenController.ChangeScreen(ScreenType.Game);
        }


    }
}
