﻿using GameResources.Manager;
using GameResources.Models.Game;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Media;
using System;
using UserInterface.Control;


namespace UserInterface.Screen
{
    public class VictoryScreen : IScreen
    {
        public ControlCollection Controls { get; }
        private readonly IScreenController screenController;

        private long lastMovementInput;
        private long lastActionInput;
        private long lastStoryUpdate;

        private string storyString;
        private int character;
        private SpriteFont storyFont;


        public VictoryScreen(IScreenController screenController)
        {
            this.Controls = new ControlCollection();
            this.screenController = screenController;

            Init();
        }


        public void Draw(SpriteBatch spriteBatch)
        {
            for (int i = 0; i < this.Controls.Count; i++)
            {
                IControl control = this.Controls[i];
                control.Draw(spriteBatch);
            }

            string text = storyString.Substring(0, character);
            spriteBatch.DrawString(storyFont, text, new Vector2(40, 40), Color.White);
        }


        public void Update(long ticks)
        {
            InputMap inputMap = screenController.GetInputMap();

            lastMovementInput += ticks;
            lastActionInput += ticks;
            lastStoryUpdate += ticks;

            //keyboard
            if (lastMovementInput >= (TimeSpan.TicksPerSecond * 0.12))
            {
                if (inputMap.Up)
                {
                    this.Controls.SelectPrev();
                    lastMovementInput = 0;
                }
                else if (inputMap.Down)
                {
                    this.Controls.SelectNext();
                    lastMovementInput = 0;
                }
            }

            if (lastActionInput >= (TimeSpan.TicksPerSecond * 0.12))
            {
                if (inputMap.Jump)
                {
                    if (this.Controls.IndexOfSelected() != -1)
                    {
                        lastActionInput = 0;
                        screenController.LastKeyboardInputTicks = 0;

                        this.Controls[this.Controls.IndexOfSelected()].Press();
                    }
                }
            }

            //story 
            if (lastStoryUpdate >= (TimeSpan.TicksPerSecond * 0.05))
            {
                character++;

                if (character > storyString.Length) character = storyString.Length;

                lastStoryUpdate = 0;
            }

            //controls
            foreach (IControl control in this.Controls)
            {
                control.Update(ticks);
            }
        }


        public void ActivateScreen()
        {
            Song song = this.screenController.GameContentManager.SongRepository.GetContentByName(@"music\sm-victory");

            if (MediaPlayer.State == MediaState.Playing)
            {
                MediaPlayer.Stop();
            }

            this.screenController.BackgroundMusic = song;
            MediaPlayer.IsRepeating = true;
            MediaPlayer.Play(song);

            lastMovementInput = screenController.LastKeyboardInputTicks;
            lastActionInput = screenController.LastKeyboardInputTicks;
        }


        private void Init()
        {
            IGameContentManager gameContentManager = this.screenController.GameContentManager;

            int buttonWidth = 220;
            int buttonHeight = 32;
            int screenWidth = screenController.Resolution.GameWidth;
            int screenHeight = screenController.Resolution.GameHeight;

            this.Controls.Add(new Button()
            {
                Text = "Main Menu",
                Location = new Rectangle(screenWidth - buttonWidth - 40, screenHeight - buttonHeight - 40, buttonWidth, buttonHeight),
                TextureNormal = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button"),
                TextureSelected = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-select"),
                TexturePressed = gameContentManager.TextureRepository.GetContentByName(@"textures\ui\button-press"),
                Font = gameContentManager.FontRepository.GetContentByName(@"fonts\Normal"),
                PressSound = gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Samus Land"),
                IsSelected = true,
            });
            ((Button)this.Controls[this.Controls.Count - 1]).OnClick += MainMenu;

            this.storyFont = gameContentManager.FontRepository.GetContentByName(@"fonts\Normal");
            character = 0;
            storyString = "";
            storyString += "YOU DID IT! CONGRATULATIONS!\n\n";
            storyString += "SAMUS RETURNED BACK TO GALACTIC CIVILIZATION!\n\n";
            storyString += "\n\n";
            storyString += "\n\n";
            storyString += "MISSION SUCCESSFUL! SEE YOU NEXT TIME!\n\n";
        }


        private void MainMenu(object sender)
        {
            this.screenController.ChangeScreen(ScreenType.MainMenu);
        }
    }
}
