﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Logic;
using GameMakerAvalonia.Models;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;

namespace GameMakerAvalonia.Dialogs
{
    public class DestructableBlockSelectDialog : Window
    {
        public ButtonResult DialogResult { get; set; }
        public DestructableBlockDefinition SelectedDestructableBlock { get; set; }

        private Session session;


        public DestructableBlockSelectDialog()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.DialogResult = ButtonResult.Cancel;
            this.FindControl<Button>("AcceptButton").Click += AcceptButton_Click;
            this.FindControl<Button>("CancelButton").Click += CancelButton_Click;
            this.FindControl<ComboBox>("DestructableBlockComboBox").SelectionChanged += DestructableBlockComboBox_SelectedIndexChanged;
        }


        public void SetParameters(Session session, int blockKey)
        {
            this.session = session;

            LoadComboBoxes();

            DestructableBlockDefinition block = session.RawContentManager.DestructableBlockRepository.GetContent(blockKey);
            
            if (block != null)
            {
                ComboBox destructableBlockComboBox = this.FindControl<ComboBox>("DestructableBlockComboBox");
                destructableBlockComboBox.SelectedItem = block;
            }
        }


        private void LoadComboBoxes()
        {
            ComboBox destructableBlockComboBox = this.FindControl<ComboBox>("DestructableBlockComboBox");

            IList<DestructableBlockDefinition> blocks = this.session.RawContentManager.DestructableBlockRepository.GetListOfContent();
            destructableBlockComboBox.Items = blocks;

            if (blocks.Count != 0)
            {
                destructableBlockComboBox.SelectedIndex = 0;
            }
        }


        private void DestructableBlockComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox destructableBlockComboBox = this.FindControl<ComboBox>("DestructableBlockComboBox");
            Image tileSetImage = this.FindControl<Image>("TileSetImage");
            Image revealTileSetImage = this.FindControl<Image>("RevealTileSetImage");

            if (this.session == null) return;
            if (destructableBlockComboBox.SelectedItem == null) return;

            DestructableBlockDefinition block = (DestructableBlockDefinition)destructableBlockComboBox.SelectedItem;

            SetImage(block.TileSetKey, tileSetImage);
            SetImage(block.RevealTileSetKey, revealTileSetImage);
        }


        private void SetImage(int tileSetKey, Image image)
        {
            TileSetDefinition tileSet = this.session.RawContentManager.TileSetRepository.GetContent(tileSetKey);
            RawResource rawResource = this.session.RawContentManager.TextureRepository.GetContent(tileSet.TextureKey);

            string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + rawResource.SystemFilePath + ".png";
            image.Source = ImageManip.GetImageSubset(filename, tileSet.X, tileSet.Y, tileSet.Width, tileSet.Height);
        }


        private void CancelButton_Click(object sender, RoutedEventArgs e)
        {
            this.DialogResult = ButtonResult.Cancel;
            this.Hide();
        }


        private async void AcceptButton_Click(object sender, RoutedEventArgs e)
        {
            ComboBox destructableBlockComboBox = this.FindControl<ComboBox>("DestructableBlockComboBox");

            if (destructableBlockComboBox.SelectedItem == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Nothing Selected",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog(this);

                return;
            }

            this.SelectedDestructableBlock = (DestructableBlockDefinition)destructableBlockComboBox.SelectedItem;

            this.DialogResult = ButtonResult.Ok;
            this.Close();
        }


    }
}
