﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media.Imaging;
using GameMakerAvalonia.Models;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;

namespace GameMakerAvalonia.Dialogs
{
    public class EnemySelectDialog : Window
    {
        public ButtonResult DialogResult { get; set; }
        public EnemyDefinition SelectedEnemy { get; set; }

        private Session session;


        public EnemySelectDialog()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.DialogResult = ButtonResult.Cancel;
            this.FindControl<Button>("AcceptButton").Click += AcceptButton_Click;
            this.FindControl<Button>("CancelButton").Click += CancelButton_Click;
            this.FindControl<ComboBox>("EnemyComboBox").SelectionChanged += EnemyComboBox_SelectedIndexChanged;
        }


        public void SetParameters(Session session, int enemyKey)
        {
            this.session = session;

            LoadComboBoxes();

            EnemyDefinition enemy = session.RawContentManager.EnemyRepository.GetContent(enemyKey);

            if (enemy != null)
            {
                ComboBox enemyComboBox = this.FindControl<ComboBox>("EnemyComboBox");
                enemyComboBox.SelectedItem = enemy;
            }
        }


        private void LoadComboBoxes()
        {
            ComboBox enemyComboBox = this.FindControl<ComboBox>("EnemyComboBox");

            IList<EnemyDefinition> enemies = this.session.RawContentManager.EnemyRepository.GetListOfContent();
            enemyComboBox.Items = enemies;

            if (enemies.Count != 0)
            {
                enemyComboBox.SelectedIndex = 0;
            }
        }


        private void EnemyComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox enemyComboBox = this.FindControl<ComboBox>("EnemyComboBox");
            Image textureImage = this.FindControl<Image>("TextureImage");

            if (this.session == null) return;
            if (enemyComboBox.SelectedItem == null) return;

            EnemyDefinition enemy = (EnemyDefinition)enemyComboBox.SelectedItem;

            SetImage(enemy, textureImage);
        }


        private void SetImage(EnemyDefinition enemy, Image image)
        {
            RawResource texture = this.session.RawContentManager.TextureRepository.GetContent(enemy.TextureKey1);

            if (texture != null)
            {
                string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + texture.SystemFilePath + ".png";
                image.Source = new Bitmap(filename);
            }
        }


        private void CancelButton_Click(object sender, RoutedEventArgs e)
        {
            this.DialogResult = ButtonResult.Cancel;
            this.Hide();
        }


        private async void AcceptButton_Click(object sender, RoutedEventArgs e)
        {
            ComboBox enemyComboBox = this.FindControl<ComboBox>("EnemyComboBox");

            if (enemyComboBox.SelectedItem == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Nothing Selected",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog(this);

                return;
            }

            this.SelectedEnemy = (EnemyDefinition)enemyComboBox.SelectedItem;

            this.DialogResult = ButtonResult.Ok;
            this.Close();
        }


    }
}
