﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media.Imaging;
using GameMakerAvalonia.Models;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;

namespace GameMakerAvalonia.Dialogs
{
    public class TextureSelectDialog : Window
    {
        public ButtonResult DialogResult { get; set; }
        public RawResource SelectedTexture { get; set; }

        private Session session;


        public TextureSelectDialog()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.DialogResult = ButtonResult.Cancel;
            this.FindControl<Button>("AcceptButton").Click += AcceptButton_Click;
            this.FindControl<Button>("CancelButton").Click += CancelButton_Click;
            this.FindControl<ComboBox>("TextureComboBox").SelectionChanged += TextureComboBox_SelectedIndexChanged;
        }


        public void SetParameters(Session session, int textureKey)
        {
            this.session = session;

            LoadComboBoxes();

            RawResource texture = session.RawContentManager.TextureRepository.GetContent(textureKey);

            if (texture != null)
            {
                ComboBox textureComboBox = this.FindControl<ComboBox>("TextureComboBox");
                textureComboBox.SelectedItem = texture;
            }
        }


        private void LoadComboBoxes()
        {
            ComboBox textureComboBox = this.FindControl<ComboBox>("TextureComboBox");

            IList<RawResource> textures = this.session.RawContentManager.TextureRepository.GetListOfContent();
            textureComboBox.Items = textures;

            if (textures.Count != 0)
            {
                textureComboBox.SelectedIndex = 0;
            }
        }


        private void TextureComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox textureComboBox = this.FindControl<ComboBox>("TextureComboBox");
            Image textureImage = this.FindControl<Image>("TextureImage");

            if (this.session == null) return;
            if (textureComboBox.SelectedItem == null) return;

            RawResource texture = (RawResource)textureComboBox.SelectedItem;

            SetImage(texture, textureImage);
        }


        private void SetImage(RawResource texture, Image image)
        {
            string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + texture.SystemFilePath + ".png";
            image.Source = new Bitmap(filename);
        }


        private void CancelButton_Click(object sender, RoutedEventArgs e)
        {
            this.DialogResult = ButtonResult.Cancel;
            this.Hide();
        }


        private async void AcceptButton_Click(object sender, RoutedEventArgs e)
        {
            ComboBox textureComboBox = this.FindControl<ComboBox>("TextureComboBox");

            if (textureComboBox.SelectedItem == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Nothing Selected",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog(this);

                return;
            }

            this.SelectedTexture = (RawResource)textureComboBox.SelectedItem;

            this.DialogResult = ButtonResult.Ok;
            this.Close();
        }


    }
}
