﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Models;
using GameMakerAvalonia.PreviewPanels;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameMakerAvalonia.Dialogs
{
    public class TileSetSelectDialog : Window
    {
        public ButtonResult DialogResult { get; set; }
        public TileSetDefinition SelectedTileSet { get; set; }

        private Session session;
        

        public TileSetSelectDialog()
        {
            this.InitializeComponent();

            this.DialogResult = ButtonResult.Cancel;
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.FindControl<Button>("AcceptButton").Click += AcceptButton_Click;
            this.FindControl<Button>("CancelButton").Click += CancelButton_Click;
            this.FindControl<ComboBox>("TextureComboBox").SelectionChanged += TextureComboBox_SelectedIndexChanged;
        }


        public void SetParameters(Session session, int tileSetKey)
        {
            this.session = session;

            LoadComboBoxes();

            TileSetDefinition tileSet = session.RawContentManager.TileSetRepository.GetContent(tileSetKey);
            int textureKey = -1;

            if (tileSet != null)
            {
                textureKey = tileSet.TextureKey;
                RawResource texture = session.RawContentManager.TextureRepository.GetContent(textureKey);

                var textureComboBox = this.FindControl<ComboBox>("TextureComboBox");
                textureComboBox.SelectedItem = texture;
            }

            var tileSetPreviewPanel = this.FindControl<TileSetPreviewPanel>("TileSetPreviewPanel");
            tileSetPreviewPanel.SetParameters(session, textureKey);
            tileSetPreviewPanel.SelectedTileSet = tileSet;
        }


        private void LoadComboBoxes()
        {
            var textureComboBox = this.FindControl<ComboBox>("TextureComboBox");

            //tile textures
            IList<RawResource> tileTextures = new List<RawResource>();

            foreach (RawResource texture in this.session.RawContentManager.TextureRepository.GetListOfContent())
            {
                bool isUsedByTileset = false;

                foreach (TileSetDefinition tileSet in this.session.RawContentManager.TileSetRepository.GetListOfContent())
                {
                    if (tileSet.TextureKey == texture.Key)
                    {
                        isUsedByTileset = true;
                        break;
                    }
                }

                if (isUsedByTileset)
                {
                    tileTextures.Add(texture);
                }
            }

            textureComboBox.Items = tileTextures;

            if (tileTextures.Count != 0)
            {
                textureComboBox.SelectedIndex = 0;
            }
        }


        private void TextureComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            var textureComboBox = this.FindControl<ComboBox>("TextureComboBox");
            var tileSetPreviewPanel = this.FindControl<TileSetPreviewPanel>("TileSetPreviewPanel");

            if (this.session == null) return;
            if (textureComboBox.SelectedItem == null) return;

            int textureKey = ((RawResource)textureComboBox.SelectedItem).Key;

            tileSetPreviewPanel.SetParameters(this.session, textureKey);
            tileSetPreviewPanel.InvalidateVisual();
        }


        private void CancelButton_Click(object sender, RoutedEventArgs e)
        {
            this.DialogResult = ButtonResult.Cancel;
            this.Hide();
        }


        private async void AcceptButton_Click(object sender, RoutedEventArgs e)
        {
            var tileSetPreviewPanel = this.FindControl<TileSetPreviewPanel>("TileSetPreviewPanel");

            if (tileSetPreviewPanel.SelectedTileSet == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Nothing Selected",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog(this);

                return;
            }

            this.SelectedTileSet = tileSetPreviewPanel.SelectedTileSet;

            this.DialogResult = ButtonResult.Ok;
            this.Close();
        }


    }
}
