﻿using Avalonia;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Models;
using GameMakerAvalonia.PreviewPanels;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;


namespace GameMakerAvalonia.Dialogs
{
    public class WorldPositionSelectDialog : Window
    {
        private Session session;
        public WorldPosition SelectedWorldPosition { get; set; }
        public ButtonResult DialogResult { get; set; }


        public WorldPositionSelectDialog()
        {
            this.InitializeComponent();
        }

        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.FindControl<Button>("AcceptButton").Click += AcceptButton_Click;
            this.FindControl<Button>("CancelButton").Click += CancelButton_Click;
            this.FindControl<ComboBox>("WorldComboBox").SelectionChanged += WorldComboBox_SelectedIndexChanged;

            WorldPreviewPanel worldCellSelect = this.FindControl<WorldPreviewPanel>("WorldCellSelect");
            worldCellSelect.PropertyChanged += WorldCellSelect_PropertyChanged;
        }


        private void CancelButton_Click(object sender, RoutedEventArgs e)
        {
            this.DialogResult = ButtonResult.Cancel;
            this.Hide();
        }

        private async void AcceptButton_Click(object sender, RoutedEventArgs e)
        {
            ComboBox worldComboBox = this.FindControl<ComboBox>("WorldComboBox");
            WorldCellPreviewPanel worldTileSelect = this.FindControl<WorldCellPreviewPanel>("WorldTileSelect");
            WorldPreviewPanel worldCellSelect = this.FindControl<WorldPreviewPanel>("WorldCellSelect");

            if ((worldTileSelect.SelectedTileX == null) || (worldCellSelect.SelectedCell == null))
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Nothing Selected",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog(this);
                
                return;
            }

            WorldMapDefinition world = (WorldMapDefinition)worldComboBox.SelectedItem;
            this.SelectedWorldPosition = new WorldPosition()
            {
                WorldMapKey = world.Key,
                WorldCellX = worldCellSelect.SelectedCell.WorldMapX,
                WorldCellY = worldCellSelect.SelectedCell.WorldMapY,
                PositionX = worldTileSelect.SelectedTileX.Value,
                PositionY = worldTileSelect.SelectedTileY.Value
            };

            this.DialogResult = ButtonResult.Ok;
            this.Close();
        }


        public void SetSession(Session session)
        {
            this.session = session;

            WorldPreviewPanel worldCellSelect = this.FindControl<WorldPreviewPanel>("WorldCellSelect");
            worldCellSelect.SetSession(session);

            WorldCellPreviewPanel worldTileSelect = this.FindControl<WorldCellPreviewPanel>("WorldTileSelect");
            worldTileSelect.SetSession(session);

            Load();
        }


        private void Load()
        {
            ComboBox worldComboBox = this.FindControl<ComboBox>("WorldComboBox");
            WorldPreviewPanel worldCellSelect = this.FindControl<WorldPreviewPanel>("WorldCellSelect");
            WorldCellPreviewPanel worldTileSelect = this.FindControl<WorldCellPreviewPanel>("WorldTileSelect");

            worldComboBox.Items = this.session.RawContentManager.WorldMapRepository.GetListOfContent();

            worldCellSelect.CellSize = 16;

            worldTileSelect.SelectedEditMode = WorldCellPreviewPanel.EditMode.ReadOnly;
            worldTileSelect.TileSize = 16;

            if (this.SelectedWorldPosition != null)
            {
                worldComboBox.SelectedItem = this.SelectedWorldPosition;

                WorldMapDefinition world = this.session.RawContentManager.WorldMapRepository.GetContent(this.SelectedWorldPosition.WorldMapKey);
                WorldMapCellDefinition cell = world.Cells[this.SelectedWorldPosition.WorldCellX][this.SelectedWorldPosition.WorldCellY];

                worldCellSelect.SelectedCell = cell;
                worldTileSelect.SetWorldMapCell(cell);

                worldTileSelect.SelectedTileX = (int)this.SelectedWorldPosition.PositionX;
                worldTileSelect.SelectedTileY = (int)this.SelectedWorldPosition.PositionY;
            }
        }


        private void WorldComboBox_SelectedIndexChanged(object sender, SelectionChangedEventArgs e)
        {
            ComboBox worldComboBox = this.FindControl<ComboBox>("WorldComboBox");
            WorldPreviewPanel worldCellSelect = this.FindControl<WorldPreviewPanel>("WorldCellSelect");
            WorldCellPreviewPanel worldTileSelect = this.FindControl<WorldCellPreviewPanel>("WorldTileSelect");

            WorldMapDefinition world = (WorldMapDefinition)worldComboBox.SelectedItem;

            worldCellSelect.SetWorldMap(world);
            worldCellSelect.SelectedCell = null;
            worldCellSelect.InvalidateVisual();

            worldTileSelect.SetWorldMapCell(null);
            worldTileSelect.SelectedTileX = null;
            worldTileSelect.SelectedTileY = null;
            worldTileSelect.InvalidateVisual();
        }


        private void WorldCellSelect_PropertyChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if(e.Property.Name.Equals("SelectedCell"))
            {
                WorldPreviewPanel worldCellSelect = this.FindControl<WorldPreviewPanel>("WorldCellSelect");
                WorldCellPreviewPanel worldTileSelect = this.FindControl<WorldCellPreviewPanel>("WorldTileSelect");

                WorldMapCellDefinition cell = worldCellSelect.SelectedCell;
                worldTileSelect.SetWorldMapCell(cell);
                worldTileSelect.SelectedTileX = null;
                worldTileSelect.SelectedTileY = null;
                worldTileSelect.InvalidateVisual();
            }
        }

    }
}
