﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Logic;
using GameMakerAvalonia.Models;
using GameMakerAvalonia.Panels;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using System;
using System.ComponentModel;
using System.IO;
using System.Threading.Tasks;

namespace GameMakerAvalonia.Forms
{
    public class MainWindow : Window
    {
        private Session session;


        public MainWindow()
        {
            InitializeComponent();
        }

        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.Closing += MainWindow_Closing;
            this.FindControl<MenuItem>("OpenMenuItem").Click += OpenMenuItem_Click;
            this.FindControl<MenuItem>("SaveMenuItem").Click += SaveMenuItem_Click;
            this.FindControl<MenuItem>("CloseMenuItem").Click += CloseMenuItem_Click;
            this.FindControl<MenuItem>("ExitMenuItem").Click += ExitMenuItem_Click;

            CloseDirectory();

#if DEBUG
            char d = Path.DirectorySeparatorChar;
            string find = d + "GameMakerAvalonia" + d + "bin" + d + "Debug" + d + "net5.0";

            string dir = Directory.GetCurrentDirectory().Replace(find, "") + d + "RawResources";
            OpenDirectory(dir);
#endif
        }


        private async void MainWindow_Closing(object sender, CancelEventArgs e)
        {
            e.Cancel = true;

            await AskToExitApp();
        }


        private async void OpenMenuItem_Click(object sender, RoutedEventArgs e)
        {
            var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
            {
                ButtonDefinitions = ButtonEnum.YesNo,
                ContentTitle = "Confirm",
                ContentMessage = "Are you sure you want to open a new session?",
                Style = Style.Windows,
                Icon = MessageBox.Avalonia.Enums.Icon.Warning
            });
            ButtonResult result = await dialog.ShowDialog(this);

            if (result == ButtonResult.No) return;

            OpenFolderDialog folderDialog = new OpenFolderDialog();
            string directory = await folderDialog.ShowAsync(this);

            if(directory.Equals("") == false)
            {
                OpenDirectory(directory);
            }
        }


        private async void SaveMenuItem_Click(object sender, RoutedEventArgs e)
        {
            var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
            {
                ButtonDefinitions = ButtonEnum.YesNo,
                ContentTitle = "Confirm",
                ContentMessage = "Are you sure you want to save the session?",
                Style = Style.Windows,
                Icon = MessageBox.Avalonia.Enums.Icon.Warning
            });
            ButtonResult result = await dialog.ShowDialog(this);

            if (result == ButtonResult.No) return;

            if (this.session == null)
            {
                dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "No session to save",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog(this);
                return;
            }

            bool status = this.session.Save();

            if (status == true)
            {
                dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Success",
                    ContentMessage = "Session Saved",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Info
                });
                await dialog.ShowDialog(this);
                return;
            }
            else
            {
                dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Save error",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog(this);
                return;
            }
        }


        private async void CloseMenuItem_Click(object sender, RoutedEventArgs e)
        {
            var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
            {
                ButtonDefinitions = ButtonEnum.YesNo,
                ContentTitle = "Confirm",
                ContentMessage = "Are you sure you want to close the session?",
                Style = Style.Windows,
                Icon = MessageBox.Avalonia.Enums.Icon.Warning
            });
            ButtonResult result = await dialog.ShowDialog(this);

            if (result == ButtonResult.No) return;

            CloseDirectory();
        }


        private async void ExitMenuItem_Click(object sender, RoutedEventArgs e)
        {
            await AskToExitApp();
        }


        private async Task AskToExitApp()
        {
            var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
            {
                ButtonDefinitions = ButtonEnum.YesNo,
                ContentTitle = "Confirm",
                ContentMessage = "Are you sure you want to exit?          ",
                Style = Style.Windows,
                Icon = MessageBox.Avalonia.Enums.Icon.None
            });
            ButtonResult result = await dialog.ShowDialog(this);

            if (result == ButtonResult.Yes)
            {
                this.Closing -= MainWindow_Closing;
                this.Close();
            }
        }


        private void CloseDirectory()
        {
            SetAppState(false);

            this.session = null;
        }


        private void OpenDirectory(string folder)
        {
            SessionLoader loader = new SessionLoader();
            this.session = loader.Load(folder);

            this.FindControl<FontsPanel>("FontsPanel").SetSession(this.session);
            this.FindControl<TexturesPanel>("TexturesPanel").SetSession(this.session);
            this.FindControl<SongsPanel>("SongsPanel").SetSession(this.session);
            this.FindControl<SoundEffectsPanel>("SoundEffectsPanel").SetSession(this.session);
            this.FindControl<TileSetsPanel>("TileSetsPanel").SetSession(this.session);
            this.FindControl<TileSetsAnimationPanel>("TileSetsAnimationPanel").SetSession(this.session);
            this.FindControl<WorldsPanel>("WorldsPanel").SetSession(this.session);
            this.FindControl<DestructableBlocksPanel>("DestructableBlocksPanel").SetSession(this.session);
            this.FindControl<PlayerPanel>("PlayerPanel").SetSession(this.session);
            this.FindControl<EnemiesPanel>("EnemiesPanel").SetSession(this.session);
            
            SetAppState(true);
        }


        private void SetAppState(bool state)
        {
            this.FindControl<MenuItem>("OpenMenuItem").IsEnabled = !state;
            this.FindControl<MenuItem>("SaveMenuItem").IsEnabled = state;
            this.FindControl<MenuItem>("CloseMenuItem").IsEnabled = state;
            this.FindControl<TabControl>("Tabs").IsEnabled = state;
        }






    }
}
