﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Models;
using GameMakerAvalonia.PreviewPanels;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameMakerAvalonia.Forms
{
    public class WorldForm : Window
    {
        private Session session;
        private WorldMapDefinition worldMap;


        public WorldForm()
        {
            this.InitializeComponent();
        }

        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.FindControl<Button>("CreateCellButton").Click += CreateCellButton_Click;
            this.FindControl<Button>("DeleteCellButton").Click += DeleteCellButton_Click;

            LoadComboBox("CellWidth");
            LoadComboBox("CellHeight");

            var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");
            worldPreviewPanel.AllowDoubleClickToEdit = true;
        }


        private void LoadComboBox(string name)
        {
            var comboBox = this.FindControl<ComboBox>(name);

            IList<int> list = new List<int>();
            for(int i = 1; i <= 10; i++)
            {
                list.Add(i);
            }

            comboBox.Items = list;

            comboBox.SelectedItem = 1;
        }


        public void SetSession(Session session)
        {
            this.session = session;

            var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");
            worldPreviewPanel.SetSession(this.session);
        }


        public void SetWorldMap(WorldMapDefinition worldMap)
        {
            this.worldMap = worldMap;

            var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");
            worldPreviewPanel.SetWorldMap(this.worldMap);
        }


        private async void CreateCellButton_Click(object sender, RoutedEventArgs e)
        {
            var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");
            var cellWidth = this.FindControl<ComboBox>("CellWidth");
            var cellHeight = this.FindControl<ComboBox>("CellHeight");

            if (worldPreviewPanel.SelectedCell == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "No position chosen",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);

                return;
            }

            WorldMapCellDefinition selectedPosition = worldPreviewPanel.SelectedCell;

            if (this.worldMap.GetCellAtLocation(selectedPosition.WorldMapX, selectedPosition.WorldMapY) != null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Cell exists at location",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);
                return;
            }

            WorldMapCellDefinition cell = new WorldMapCellDefinition(this.worldMap.Key, selectedPosition.WorldMapX, selectedPosition.WorldMapY, (int)cellWidth.SelectedItem, (int)cellHeight.SelectedItem, -1, null);
            this.worldMap.AddCell(cell);

            worldPreviewPanel.SelectedCell = null;
            worldPreviewPanel.InvalidateVisual();
        }


        private async void DeleteCellButton_Click(object sender, RoutedEventArgs e)
        {
            var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");

            if (worldPreviewPanel.SelectedCell == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "No position chosen",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);

                return;
            }

            WorldMapCellDefinition selectedPosition = worldPreviewPanel.SelectedCell;
            WorldMapCellDefinition selectedCell = this.worldMap.GetCellAtLocation(selectedPosition.WorldMapX, selectedPosition.WorldMapY);

            if (selectedCell == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "No cell exists at location",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);

                return;
            }

            var confirmDialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
            {
                ButtonDefinitions = ButtonEnum.YesNo,
                ContentTitle = "Confirm",
                ContentMessage = "Are you sure?",
                Style = Style.Windows,
                Icon = MessageBox.Avalonia.Enums.Icon.Warning
            });
            ButtonResult result = await confirmDialog.ShowDialog((Window)this.VisualRoot);

            if (result == ButtonResult.Yes)
            {
                this.worldMap.RemoveCell(selectedCell);

                worldPreviewPanel.SelectedCell = null;
                worldPreviewPanel.InvalidateVisual();
            }
        }


    }
}
