﻿using Avalonia;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Dialogs;
using GameMakerAvalonia.Logic;
using GameMakerAvalonia.Models;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;

namespace GameMakerAvalonia.Panels
{
    public class DestructableBlocksPanel : UserControl
    {
        private Session session;
        private IList<DestructableBlockDefinition> resources;


        public DestructableBlocksPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            LoadComboBoxes();

            AddEvents();

            this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
        }


        private void AddEvents()
        {
            this.FindControl<Button>("RevealTileSetSelectButton").Click += RevealTileSetSelectButton_Click;
            this.FindControl<Button>("TileSetSelectButton").Click += TileSetSelectButton_Click;
            this.FindControl<Button>("AddBlock").Click += AddBlock_Click;
            this.FindControl<Button>("DeleteBlock").Click += DeleteBlock_Click;
            this.FindControl<ComboBox>("DamageType").SelectionChanged += DamageType_SelectedChanged;
            this.FindControl<TextBox>("NameText").PropertyChanged += NameText_TextChanged;
            this.FindControl<TextBox>("OnlyShowIfVariableIsFalseText").PropertyChanged += OnlyShowIfVariableIsFalseText_TextChanged;
            this.FindControl<TextBox>("SetVariableToTrueUponDestructionText").PropertyChanged += SetVariableToTrueUponDestructionText_TextChanged;
        }


        private void RemoveEvents()
        {
            this.FindControl<Button>("RevealTileSetSelectButton").Click -= RevealTileSetSelectButton_Click;
            this.FindControl<Button>("TileSetSelectButton").Click -= TileSetSelectButton_Click;
            this.FindControl<Button>("AddBlock").Click -= AddBlock_Click;
            this.FindControl<Button>("DeleteBlock").Click -= DeleteBlock_Click;
            this.FindControl<ComboBox>("DamageType").SelectionChanged -= DamageType_SelectedChanged;
            this.FindControl<TextBox>("NameText").PropertyChanged -= NameText_TextChanged;
            this.FindControl<TextBox>("OnlyShowIfVariableIsFalseText").PropertyChanged -= OnlyShowIfVariableIsFalseText_TextChanged;
            this.FindControl<TextBox>("SetVariableToTrueUponDestructionText").PropertyChanged -= SetVariableToTrueUponDestructionText_TextChanged;
        }


        public void SetSession(Session session)
        {
            this.session = session;

            LoadSession();
        }


        private void LoadSession()
        {
            this.resources = session.RawContentManager.DestructableBlockRepository.GetListOfContent();

            var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;

            RefreshForm();
        }


        private void LoadComboBoxes()
        {
            ComboBox damageType = this.FindControl<ComboBox>("DamageType");
            damageType.Items = Enum.GetValues(typeof(DamageType));
        }



        private void ListGridView_CurrentCellChanged(object sender, EventArgs e)
        {
            RefreshForm();
        }


        private void RefreshForm()
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            Image tileSetPreview = this.FindControl<Image>("TileSetPreview");
            Image revealTileSetPreview = this.FindControl<Image>("RevealTileSetPreview");
            ComboBox damageType = this.FindControl<ComboBox>("DamageType");
            TextBox nameText = this.FindControl<TextBox>("NameText");
            TextBox onlyShowIfVariableIsFalseText = this.FindControl<TextBox>("OnlyShowIfVariableIsFalseText");
            TextBox setVariableToTrueUponDestructionText = this.FindControl<TextBox>("SetVariableToTrueUponDestructionText");

            RemoveEvents();

            if (grid.SelectedItem != null)
            {
                DestructableBlockDefinition item = (DestructableBlockDefinition)grid.SelectedItem;

                TileSetDefinition tileSet = this.session.RawContentManager.TileSetRepository.GetContent(item.TileSetKey);
                if (tileSet != null)
                {
                    RawResource rawResource = this.session.RawContentManager.TextureRepository.GetContent(tileSet.TextureKey);

                    string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + rawResource.SystemFilePath + ".png";
                    tileSetPreview.Source = ImageManip.GetImageSubset(filename, tileSet.X, tileSet.Y, tileSet.Width, tileSet.Height);
                }

                TileSetDefinition revealTileSet = this.session.RawContentManager.TileSetRepository.GetContent(item.RevealTileSetKey);
                if (revealTileSet != null)
                {
                    RawResource rawResource = this.session.RawContentManager.TextureRepository.GetContent(revealTileSet.TextureKey);

                    string filename = this.session.WorkingDirectory + Path.DirectorySeparatorChar + "Content" + Path.DirectorySeparatorChar + rawResource.SystemFilePath + ".png";
                    revealTileSetPreview.Source = ImageManip.GetImageSubset(filename, revealTileSet.X, revealTileSet.Y, revealTileSet.Width, revealTileSet.Height);
                }

                damageType.SelectedItem = item.DamageType;
                
                nameText.Text = item.Name;
                onlyShowIfVariableIsFalseText.Text = item.OnlyShowIfVariableIsFalse;
                setVariableToTrueUponDestructionText.Text = item.SetVariableToTrueUponDestruction;
            }
            else
            {
                tileSetPreview.Source = null;
                revealTileSetPreview.Source = null;
                damageType.SelectedItem = null;
                nameText.Text = "";
                onlyShowIfVariableIsFalseText.Text = "";
                setVariableToTrueUponDestructionText.Text = "";
            }

            AddEvents();
        }


        private void AddBlock_Click(object sender, RoutedEventArgs e)
        {
            DestructableBlockDefinition block = new DestructableBlockDefinition();

            this.session.RawContentManager.DestructableBlockRepository.AddContent(block);

            var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = null;

            LoadSession();
        }


        private async void DeleteBlock_Click(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Select a block",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);

                return;
            }

            DestructableBlockDefinition block = (DestructableBlockDefinition)grid.SelectedItem;

            this.session.RawContentManager.DestructableBlockRepository.RemoveContent(block.Key);

            grid.Items = null;

            LoadSession();
        }


        private async void TileSetSelectButton_Click(object s, RoutedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                DestructableBlockDefinition block = (DestructableBlockDefinition)grid.SelectedItem;

                TileSetSelectDialog dialog = new TileSetSelectDialog();
                dialog.SetParameters(session, block.TileSetKey);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == ButtonResult.Ok)
                {
                    block.TileSetKey = dialog.SelectedTileSet.Key;

                    RefreshForm();
                }
            }
        }


        private async void RevealTileSetSelectButton_Click(object s, RoutedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                DestructableBlockDefinition block = (DestructableBlockDefinition)grid.SelectedItem;

                TileSetSelectDialog dialog = new TileSetSelectDialog();
                dialog.SetParameters(session, block.TileSetKey);

                await dialog.ShowDialog((Window)this.VisualRoot);

                if (dialog.DialogResult == ButtonResult.Ok)
                {
                    block.RevealTileSetKey = dialog.SelectedTileSet.Key;

                    RefreshForm();
                }
            }
        }


        private void DamageType_SelectedChanged(object sender, SelectionChangedEventArgs e)
        {
            ComboBox damageType = this.FindControl<ComboBox>("DamageType");
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            if(grid.SelectedItem != null)
            {
                DestructableBlockDefinition block = (DestructableBlockDefinition)grid.SelectedItem;

                block.DamageType = (DamageType)damageType.SelectedItem;
            }
        }


        private void NameText_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                if (session == null) return;

                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    DestructableBlockDefinition item = (DestructableBlockDefinition)grid.SelectedItem;
                    TextBox textBox = this.FindControl<TextBox>("NameText");

                    item.Name = textBox.Text;
                }
            }
        }


        private void OnlyShowIfVariableIsFalseText_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                if (session == null) return;

                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    DestructableBlockDefinition item = (DestructableBlockDefinition)grid.SelectedItem;
                    TextBox textBox = this.FindControl<TextBox>("OnlyShowIfVariableIsFalseText");

                    item.OnlyShowIfVariableIsFalse = textBox.Text;
                }
            }
        }


        private void SetVariableToTrueUponDestructionText_TextChanged(object sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (e.Property.Name.Equals("Text"))
            {
                if (session == null) return;

                var grid = this.FindControl<DataGrid>("ListGridView");

                if (grid.SelectedItem != null)
                {
                    DestructableBlockDefinition item = (DestructableBlockDefinition)grid.SelectedItem;
                    TextBox textBox = this.FindControl<TextBox>("SetVariableToTrueUponDestructionText");

                    item.SetVariableToTrueUponDestruction = textBox.Text;
                }
            }
        }


    }
}
