﻿using GameResources.Models.World;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Media;
using RawResources.Models;
using RawResources.Repository;
using System.Collections.Generic;


namespace GameResources.Repository
{
    public class CachedWorldMapRepository : IGameContentRepository<WorldMap>
    {
        private readonly IRawContentRepository<WorldMapDefinition> definitionRepository;
        private readonly IGameContentRepository<TileSet> tileSetRepository;
        private readonly IGameNamedContentRepository<Song> songRepository;
        private readonly IGameNamedContentRepository<Texture2D> textureRepository;
        private readonly IDictionary<int, WorldMap> cache;


        public CachedWorldMapRepository(IRawContentRepository<WorldMapDefinition> definitionRepository, 
                                        IGameContentRepository<TileSet> tileSetRepository, 
                                        IGameNamedContentRepository<Song> songRepository,
                                        IGameNamedContentRepository<Texture2D> textureRepository)
        {
            this.definitionRepository = definitionRepository;
            this.tileSetRepository = tileSetRepository;
            this.songRepository = songRepository;
            this.textureRepository = textureRepository;
            this.cache = new Dictionary<int, WorldMap>();
        }

        public WorldMap GetContentByKey(int key)
        {
            if (this.cache.ContainsKey(key)) return this.cache[key];

            WorldMapDefinition definition = this.definitionRepository.GetContent(key);
            if (definition == null) return null;

            Load(definition);

            return this.cache[key];
        }


        private void Load(WorldMapDefinition definition)
        {
            WorldMap worldMap = new WorldMap()
            {
                Key = definition.Key,
                Name = definition.Name
            };

            foreach(WorldMapCellDefinition cellDefinition in definition.GetListOfCells())
            {
                WorldMapCell cell = new WorldMapCell()
                {
                    Song = this.songRepository.GetContentByKey(cellDefinition.SongKey),
                    WorldMapHeight = cellDefinition.WorldMapHeight,
                    WorldMapWidth = cellDefinition.WorldMapWidth,
                    WorldMapX = cellDefinition.WorldMapX,
                    WorldMapY = cellDefinition.WorldMapY,
                    WorldKey = cellDefinition.WorldKey
                };

                cell.DestructableBlockPlacements = cellDefinition.DestructableBlockPlacements;
                cell.DoorPlacements = cellDefinition.DoorPlacements;
                cell.TeleporterPlacements = cellDefinition.TeleporterPlacements;
                cell.ItemPlacements = cellDefinition.ItemPlacements;
                cell.SaverPlacements = cellDefinition.SaverPlacements;
                cell.EnemyPlacements = cellDefinition.EnemyPlacements;
                cell.VictoryPlacements = cellDefinition.VictoryPlacements;
                cell.IsHeated = cellDefinition.IsHeated;
                cell.WaterLevel = cellDefinition.WaterLevel * TileSet.DRAW_SIZE;

                if(cellDefinition.ParallaxBackgroundTextureKey != null)
                {
                    cell.ParallaxBackground = this.textureRepository.GetContentByKey(cellDefinition.ParallaxBackgroundTextureKey.Value);
                }

                for(int x = 0; x < cellDefinition.GetMaxTileX(); x++)
                {
                    for(int y = 0; y < cellDefinition.GetMaxTileY(); y++)
                    {
                        TileSet layer1 = null;
                        TileSet layer2 = null;

                        int? tile1Key = cellDefinition.TileSetKeysLayer1[x][y];
                        if (tile1Key == null) tile1Key = -1;
                        layer1 = this.tileSetRepository.GetContentByKey(tile1Key.Value);

                        int? tile2Key = cellDefinition.TileSetKeysLayer2[x][y];
                        if (tile2Key == null) tile2Key = -1;
                        layer2 = this.tileSetRepository.GetContentByKey(tile2Key.Value);


                        cell.AddTile(x, y, layer1, layer2);
                    }
                }

                worldMap.AddCell(cell);
            }

            this.cache.Add(worldMap.Key, worldMap);
        }


    }
}
