﻿using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;


namespace RawResources.Repository
{
    public class FileDestructableBlockRepository : IRawContentRepository<DestructableBlockDefinition>
    {
        private IDictionary<int, DestructableBlockDefinition> contentsByKey;
        private IList<int> keys;


        public FileDestructableBlockRepository(string filename)
        {
            Load(filename);
        }


        public DestructableBlockDefinition GetContent(int key)
        {
            DestructableBlockDefinition content = default;

            if (this.contentsByKey.ContainsKey(key))
            {
                content = this.contentsByKey[key];
            }

            return content;
        }


        public IList<DestructableBlockDefinition> GetListOfContent()
        {
            IList<DestructableBlockDefinition> content = new List<DestructableBlockDefinition>();
            IList<int> keys = GetKeys();

            foreach (int key in keys)
            {
                content.Add(contentsByKey[key]);
            }

            return content;
        }


        public void AddContent(DestructableBlockDefinition content)
        {
            int key = GetMaxKey() + 1;

            content.Key = key;

            this.contentsByKey.Add(key, content);
            this.keys.Add(key);
        }


        public void RemoveContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                this.contentsByKey.Remove(key);
                this.keys.Remove(key);
            }
        }


        private int GetMaxKey()
        {
            int maxKey = -1;

            foreach (int key in this.keys)
            {
                if (maxKey < key) maxKey = key;
            }

            return maxKey;
        }


        public IList<int> GetKeys()
        {
            return this.keys;
        }


        public void Load(string filename)
        {
            Stream stream = null;
            StreamReader reader = null;
            string line = null;
            string[] lineContents = null;

            this.contentsByKey = new Dictionary<int, DestructableBlockDefinition>();
            this.keys = new List<int>();

            try
            {
                stream = File.OpenRead(filename);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return;
            }

            //skip first line
            reader.ReadLine();

            while ((line = reader.ReadLine()) != null)
            {
                lineContents = line.Split(',');

                if (lineContents[0].Equals("") == false)
                {
                    LoadContent(lineContents);
                }
            }

            reader.Close();
        }


        private void LoadContent(string[] lineContents)
        {
            DestructableBlockDefinition definition = new DestructableBlockDefinition();

            definition.Key = Int32.Parse(lineContents[0]);
            definition.Name = lineContents[1];
            definition.TileSetKey = Int32.Parse(lineContents[2]);
            definition.RevealTileSetKey = Int32.Parse(lineContents[3]);
            definition.DamageType = (DamageType)Enum.Parse(typeof(DamageType), lineContents[4]);
            definition.OnlyShowIfVariableIsFalse = lineContents[5];
            definition.SetVariableToTrueUponDestruction = lineContents[6];

            this.contentsByKey.Add(definition.Key, definition);
            this.keys.Add(definition.Key);
        }


        public void Save(string filename)
        {
            StreamWriter writer = new StreamWriter(filename, false);

            writer.WriteLine(DestructableBlockDefinition.GetCsvTitle());

            foreach (int key in this.keys)
            {
                DestructableBlockDefinition block = this.contentsByKey[key];

                writer.WriteLine(block.ToCsvString());
            }

            writer.Close();
        }

    }
}
