﻿using RawResources.Models;
using System;
using System.Collections.Generic;
using System.IO;


namespace RawResources.Repository
{
    public class FileWorldMapRepository : IRawContentRepository<WorldMapDefinition>
    {
        private IDictionary<int, WorldMapDefinition> contentsByKey;
        private IList<int> keys;


        public FileWorldMapRepository(string filename)
        {
            Load(filename);
        }


        public WorldMapDefinition GetContent(int key)
        {
            WorldMapDefinition content = null;

            if (this.contentsByKey.ContainsKey(key))
            {
                content = this.contentsByKey[key];
            }

            return content;
        }


        public IList<WorldMapDefinition> GetListOfContent()
        {
            IList<WorldMapDefinition> content = new List<WorldMapDefinition>();
            IList<int> keys = GetKeys();

            foreach (int key in keys)
            {
                content.Add(contentsByKey[key]);
            }

            return content;
        }


        public void AddContent(WorldMapDefinition content)
        {
            int key = GetMaxKey() + 1;

            content.Key = key;
            content.Filename = "worldmap-" + key + ".csv";

            this.contentsByKey.Add(key, content);
            this.keys.Add(key);
        }


        public void RemoveContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                this.contentsByKey.Remove(key);
                this.keys.Remove(key);
            }
        }


        private int GetMaxKey()
        {
            int maxKey = -1;

            foreach (int key in this.keys)
            {
                if (maxKey < key) maxKey = key;
            }

            return maxKey;
        }


        public IList<int> GetKeys()
        {
            return this.keys;
        }


        public void Load(string filename)
        {
            Stream stream = null;
            StreamReader reader = null;
            string line = null;
            string[] lineContents = null;

            string path = GetPathFromFile(filename);
            if (path.Equals("") == false) path += Path.DirectorySeparatorChar;

            this.contentsByKey = new Dictionary<int, WorldMapDefinition>();
            this.keys = new List<int>();

            try
            {
                stream = File.OpenRead(filename);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return;
            }

            //skip first line
            reader.ReadLine();

            while ((line = reader.ReadLine()) != null)
            {
                lineContents = line.Split(',');

                if (lineContents[0].Equals("") == false)
                {
                    LoadContent(path,lineContents);
                }
            }

            reader.Close();
        }


        private void LoadContent(string path, string[] lineContents)
        {
            WorldMapDefinition worldMap = new WorldMapDefinition()
            {
                Key = Int32.Parse(lineContents[0]),
                Name = lineContents[1],
                Filename = lineContents[2]
            };

            LoadCells(path, worldMap);

            this.contentsByKey.Add(worldMap.Key, worldMap);
            this.keys.Add(worldMap.Key);
        }


        public void Save(string filename)
        {
            StreamWriter writer = new StreamWriter(filename, false);

            writer.WriteLine(WorldMapDefinition.GetCsvTitle());

            foreach (int key in this.keys)
            {
                WorldMapDefinition worldMap = this.contentsByKey[key];

                writer.WriteLine(worldMap.ToCsvString());
            }

            writer.Close();

            string path = GetPathFromFile(filename);
            if (path.Equals("") == false) path += Path.DirectorySeparatorChar;

            foreach (int key in this.keys)
            {
                WorldMapDefinition worldMap = this.contentsByKey[key];

                SaveCells(path, worldMap);
            }
        }


        private void LoadCells(string path, WorldMapDefinition worldMap)
        {
            string line = null;
            string[] lineContents = null;
            string[] layerSplit = null;
            WorldMapCellDefinition cell = null;
            int tileX = 0;
            int tileY = 0;
            int? tile1 = null;
            int? tile2 = null;

            Stream stream = File.OpenRead(path + worldMap.Filename);
            StreamReader reader = new StreamReader(stream);

            while ((line = reader.ReadLine()) != null)
            {
                if (line.StartsWith("Cell"))
                {
                    lineContents = line.Split('|');

                    cell = new WorldMapCellDefinition();
                    cell.WorldKey = worldMap.Key;
                    cell.WorldMapX = Int32.Parse(lineContents[1]);
                    cell.WorldMapY = Int32.Parse(lineContents[2]);
                    cell.WorldMapWidth = Int32.Parse(lineContents[3]);
                    cell.WorldMapHeight = Int32.Parse(lineContents[4]);
                    cell.SongKey = Int32.Parse(lineContents[5]);

                    if (lineContents[6].Equals("") == false) cell.ParallaxBackgroundTextureKey = Int32.Parse(lineContents[6]);
                    else cell.ParallaxBackgroundTextureKey = null;

                    cell.IsHeated = Boolean.Parse(lineContents[7]);
                    cell.WaterLevel = Int32.Parse(lineContents[8]);

                    worldMap.AddCell(cell);

                    tileX = 0;
                    tileY = 0;
                }
                else if (line.StartsWith("Block"))
                {
                    lineContents = line.Split('|');

                    DestructableBlockPlacementDefinition block = new DestructableBlockPlacementDefinition()
                    {
                        X = Int32.Parse(lineContents[1]),
                        Y = Int32.Parse(lineContents[2]),
                        DestructableBlockKey = Int32.Parse(lineContents[3])
                    };

                    cell.DestructableBlockPlacements.Add(block);
                }
                else if (line.StartsWith("Enemy"))
                {
                    lineContents = line.Split('|');

                    EnemyPlacementDefinition enemy = new EnemyPlacementDefinition()
                    {
                        X = Int32.Parse(lineContents[1]),
                        Y = Int32.Parse(lineContents[2]),
                        EnemyKey = Int32.Parse(lineContents[3])
                    };

                    cell.EnemyPlacements.Add(enemy);
                }
                else if (line.StartsWith("Item"))
                {
                    lineContents = line.Split('|');

                    ItemPlacementDefinition item = new ItemPlacementDefinition()
                    {
                        WorldKey = Int32.Parse(lineContents[1]),
                        CellX = Int32.Parse(lineContents[2]),
                        CellY = Int32.Parse(lineContents[3]),
                        X = Int32.Parse(lineContents[4]),
                        Y = Int32.Parse(lineContents[5]),
                        ItemType = (ItemType)Enum.Parse(typeof(ItemType), lineContents[6])
                    };
                    cell.ItemPlacements.Add(item);
                }
                else if (line.StartsWith("Door"))
                {
                    lineContents = line.Split('|');

                    DoorPlacementDefinition door = new DoorPlacementDefinition()
                    {
                        X = Int32.Parse(lineContents[1]),
                        Y = Int32.Parse(lineContents[2]),
                        DamageType = (DamageType)Enum.Parse(typeof(DamageType), lineContents[3]),
                        DoorDirection = (DoorDirection)Enum.Parse(typeof(DoorDirection), lineContents[4])
                    };

                    cell.DoorPlacements.Add(door);
                }
                else if (line.StartsWith("Saver"))
                {
                    lineContents = line.Split('|');

                    SaverDefinition saver = new SaverDefinition()
                    {
                        X = Int32.Parse(lineContents[1]),
                        Y = Int32.Parse(lineContents[2]),
                    };

                    cell.SaverPlacements.Add(saver);
                }
                else if (line.StartsWith("Victory"))
                {
                    lineContents = line.Split('|');

                    VictoryDefinition victory = new VictoryDefinition()
                    {
                        X = Int32.Parse(lineContents[1]),
                        Y = Int32.Parse(lineContents[2]),
                    };

                    cell.VictoryPlacements.Add(victory);
                }
                else if (line.StartsWith("Teleporter"))
                {
                    lineContents = line.Split('|');

                    TeleporterDefinition teleporter = new TeleporterDefinition()
                    {
                        X = Int32.Parse(lineContents[1]),
                        Y = Int32.Parse(lineContents[2]),
                    };

                    lineContents = lineContents[3].Split(',');

                    teleporter.Destination = new WorldPosition()
                    {
                        WorldMapKey = Int32.Parse(lineContents[0]),
                        WorldCellX = Int32.Parse(lineContents[1]),
                        WorldCellY = Int32.Parse(lineContents[2]),
                        PositionX = Int32.Parse(lineContents[3]),
                        PositionY = Int32.Parse(lineContents[4])
                    };

                    cell.TeleporterPlacements.Add(teleporter);
                }
                else
                {
                    lineContents = line.Split(',');

                    for (int i = 0; i < lineContents.Length; i++)
                    {
                        if (lineContents[i].Contains("|"))
                        {
                            layerSplit = lineContents[i].Split('|');

                            if (layerSplit[0].Equals("") == false) tile1 = Int32.Parse(layerSplit[0]);
                            else tile1 = null;

                            if (layerSplit[1].Equals("") == false) tile2 = Int32.Parse(layerSplit[1]);
                            else tile2 = null;
                        }
                        else if (lineContents[i].Equals(""))
                        {
                            tile1 = null;
                            tile2 = null;
                        }
                        else
                        {
                            tile1 = Int32.Parse(lineContents[i]);
                            tile2 = null;
                        }

                        cell.AddTile(tileX, tileY, tile1, tile2);

                        tileX++;
                    }

                    tileX = 0;
                    tileY++;
                }
            }

            reader.Close();
        }


        private string GetPathFromFile(string file)
        {
            int index = file.LastIndexOf(Path.DirectorySeparatorChar);

            if(index != -1)
            {
                return file.Substring(0, index);
            }
            else
            {
                return "";
            }
        }


        private void SaveCells(string path, WorldMapDefinition worldMap)
        {
            StreamWriter writer = new StreamWriter(path + worldMap.Filename, false);
            IList<WorldMapCellDefinition> cells = worldMap.GetListOfCells();

            foreach(WorldMapCellDefinition cell in cells)
            {
                writer.Write(cell.GetCSV());
            }

            writer.Close();
        }


    } 
}
