﻿using Avalonia.Controls;
using Avalonia.Input;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Models;
using GameMakerAvalonia.PreviewPanels;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameMakerAvalonia.Forms
{
    public class WorldCellForm : Window
    {
        private Session session;
        private WorldMapCellDefinition worldMapCell;


        public WorldCellForm()
        {
            this.InitializeComponent();
        }

        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.FindControl<ComboBox>("SongComboBox").SelectionChanged += SongComboBox_SelectedIndexChanged;
            this.FindControl<ComboBox>("TextureComboBox").SelectionChanged += TextureComboBox_SelectedIndexChanged;
            this.FindControl<ComboBox>("EditModeSelector").SelectionChanged += EditModeSelector_SelectedIndexChanged;
            this.FindControl<ComboBox>("ParallaxComboBox").SelectionChanged += ParallaxComboBox_SelectedIndexChanged;
            this.FindControl<ComboBox>("WaterLevelComboBox").SelectionChanged += WaterLevelComboBox_SelectedIndexChanged;
            this.FindControl<CheckBox>("IsHeatedCheckBox").Checked += IsHeatedCheckBox_Checked;
            this.FindControl<Button>("DeleteObject").Click += DeleteObject_Click;
            this.PropertyChanged += WorldCellForm_PropertyChanged;

            LoadEditModeComboBox();

            ComboBox editModeSelector = this.FindControl<ComboBox>("EditModeSelector");
            editModeSelector.SelectedItem = WorldCellPreviewPanel.EditMode.ReadOnly;

            WorldCellPreviewPanel worldCellPreviewPanel = this.FindControl<WorldCellPreviewPanel>("WorldCellPreviewPanel");
            TileSetPreviewPanel tileSetPreviewPanel = this.FindControl<TileSetPreviewPanel>("TileSetPreviewPanel");
            worldCellPreviewPanel.TileSize = 32;
            worldCellPreviewPanel.TileSetSelectionPanel = tileSetPreviewPanel;

            this.KeyUp += WorldCellForm_KeyUp;
            this.KeyDown += WorldCellForm_KeyDown;
        }


        private void IsHeatedCheckBox_Checked(object sender, RoutedEventArgs e)
        {
            CheckBox isHeatedCheckBox = this.FindControl<CheckBox>("IsHeatedCheckBox");

            this.worldMapCell.IsHeated = isHeatedCheckBox.IsChecked.Value;
        }


        private void WorldCellForm_PropertyChanged(object sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            if(e.Property.Name.Equals("ClientSize"))
            {
                ScrollViewer scrollViewer = this.FindControl<ScrollViewer>("TileSetScrollPanel");
                scrollViewer.Height = this.Height - 200;
                scrollViewer.InvalidateVisual();

                scrollViewer = this.FindControl<ScrollViewer>("WorldCellScrollPanel");
                scrollViewer.Height = this.Height - 100;
                scrollViewer.InvalidateVisual();
            }
        }


        private void WorldCellForm_KeyUp(object sender, KeyEventArgs e)
        {
            WorldCellPreviewPanel worldCellPreviewPanel = this.FindControl<WorldCellPreviewPanel>("WorldCellPreviewPanel");

            if(worldCellPreviewPanel.IsShiftDown)
            {
                if(e.Key == Key.LeftShift || e.Key == Key.RightShift)
                {
                    worldCellPreviewPanel.IsShiftDown = false;
                }
            }

            if (worldCellPreviewPanel.IsControlDown)
            {
                if (e.Key == Key.LeftCtrl || e.Key == Key.RightCtrl)
                {
                    worldCellPreviewPanel.IsControlDown = false;
                }
            }
        }


        private void WorldCellForm_KeyDown(object sender, KeyEventArgs e)
        {
            WorldCellPreviewPanel worldCellPreviewPanel = this.FindControl<WorldCellPreviewPanel>("WorldCellPreviewPanel");

            worldCellPreviewPanel.IsShiftDown = (e.Key == Key.LeftShift || e.Key == Key.RightShift);
            worldCellPreviewPanel.IsControlDown = (e.Key == Key.LeftCtrl || e.Key == Key.RightCtrl);
        }



        private void LoadEditModeComboBox()
        {
            var editModeSelector = this.FindControl<ComboBox>("EditModeSelector");

            editModeSelector.Items = Enum.GetValues(typeof(WorldCellPreviewPanel.EditMode));
        }


        public void SetSession(Session session)
        {
            this.session = session;

            WorldCellPreviewPanel worldCellPreviewPanel = this.FindControl<WorldCellPreviewPanel>("WorldCellPreviewPanel");
            TileSetPreviewPanel tileSetPreviewPanel = this.FindControl<TileSetPreviewPanel>("TileSetPreviewPanel");
            tileSetPreviewPanel.SetParameters(session, -1);
            worldCellPreviewPanel.SetSession(session);

            LoadComboBoxes();
        }


        public void SetWorldMapCell(WorldMapCellDefinition worldMapCell)
        {
            this.worldMapCell = worldMapCell;

            WorldCellPreviewPanel worldCellPreviewPanel = this.FindControl<WorldCellPreviewPanel>("WorldCellPreviewPanel");
            worldCellPreviewPanel.SetWorldMapCell(worldMapCell);

            LoadComboBoxes();
        }


        private void LoadComboBoxes()
        {
            ComboBox songComboBox = this.FindControl<ComboBox>("SongComboBox");
            ComboBox textureComboBox = this.FindControl<ComboBox>("TextureComboBox");
            ComboBox parallaxComboBox = this.FindControl<ComboBox>("ParallaxComboBox");
            ComboBox waterLevelComboBox = this.FindControl<ComboBox>("WaterLevelComboBox");
            CheckBox isHeatedCheckBox = this.FindControl<CheckBox>("IsHeatedCheckBox");

            if (this.session == null) return;
            if (this.worldMapCell == null) return;

            //song
            songComboBox.Items = this.session.RawContentManager.SongRepository.GetListOfContent();
            if (this.worldMapCell.SongKey != -1)
            {
                songComboBox.SelectedItem = this.session.RawContentManager.SongRepository.GetContent(this.worldMapCell.SongKey);
            }

            //tile textures
            IList<RawResource> tileTextures = new List<RawResource>();

            foreach (RawResource texture in this.session.RawContentManager.TextureRepository.GetListOfContent())
            {
                bool isUsedByTileset = false;

                foreach (TileSetDefinition tileSet in this.session.RawContentManager.TileSetRepository.GetListOfContent())
                {
                    if (tileSet.TextureKey == texture.Key)
                    {
                        isUsedByTileset = true;
                        break;
                    }
                }

                if (isUsedByTileset)
                {
                    tileTextures.Add(texture);
                }
            }

            textureComboBox.Items = tileTextures;

            if (tileTextures.Count != 0)
            {
                textureComboBox.SelectedIndex = 0;
            }

            //parallax textures
            IList<RawResource> parallaxTextures = new List<RawResource>();

            foreach (RawResource texture in this.session.RawContentManager.TextureRepository.GetListOfContent())
            {
                if (texture.FilePath.Contains("parallax"))
                {
                    parallaxTextures.Add(texture);
                }
            }

            parallaxComboBox.Items = parallaxTextures;

            if (this.worldMapCell.ParallaxBackgroundTextureKey != null)
            {
                parallaxComboBox.SelectedItem = this.session.RawContentManager.TextureRepository.GetContent(this.worldMapCell.ParallaxBackgroundTextureKey.Value);
            }

            //water level
            IList<int> levels = new List<int>();
            for(int i = 0; i <= this.worldMapCell.GetMaxTileY(); i++) levels.Add(i);
            waterLevelComboBox.Items = levels;
            waterLevelComboBox.SelectedItem = this.worldMapCell.WaterLevel;

            //is heated checkbox
            isHeatedCheckBox.IsChecked = this.worldMapCell.IsHeated;
        }


        private void SongComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox songComboBox = this.FindControl<ComboBox>("SongComboBox");

            if (this.worldMapCell == null) return;
            if (songComboBox.SelectedItem == null) return;

            this.worldMapCell.SongKey = ((RawResource)songComboBox.SelectedItem).Key;
        }


        private void TextureComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox textureComboBox = this.FindControl<ComboBox>("TextureComboBox");
            TileSetPreviewPanel tileSetPreviewPanel = this.FindControl<TileSetPreviewPanel>("TileSetPreviewPanel");

            if (this.worldMapCell == null) return;
            if (this.session == null) return;
            if (textureComboBox.SelectedItem == null) return;

            int textureKey = ((RawResource)textureComboBox.SelectedItem).Key;

            tileSetPreviewPanel.SetParameters(this.session, textureKey);
            tileSetPreviewPanel.InvalidateVisual();
        }


        private void EditModeSelector_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox editModeSelector = this.FindControl<ComboBox>("EditModeSelector");
            WorldCellPreviewPanel worldCellPreviewPanel = this.FindControl<WorldCellPreviewPanel>("WorldCellPreviewPanel");
            
            worldCellPreviewPanel.SelectedEditMode = (WorldCellPreviewPanel.EditMode)editModeSelector.SelectedItem;

            worldCellPreviewPanel.InvalidateVisual();
        }


        private void ParallaxComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox parallaxComboBox = this.FindControl<ComboBox>("ParallaxComboBox");
            if (this.worldMapCell == null) return;
            int textureKey = ((RawResource)parallaxComboBox.SelectedItem).Key;

            if (textureKey != -1)
            {
                this.worldMapCell.ParallaxBackgroundTextureKey = textureKey;
            }
            else
            {
                this.worldMapCell.ParallaxBackgroundTextureKey = null;
            }
        }


        private void WaterLevelComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboBox waterLevelComboBox = this.FindControl<ComboBox>("WaterLevelComboBox");

            if (this.worldMapCell == null) return;
            if (waterLevelComboBox.SelectedItem == null) return;

            int level = (int)waterLevelComboBox.SelectedItem;

            this.worldMapCell.WaterLevel = level;
        }


        private void DeleteObject_Click(object sender, RoutedEventArgs e)
        {
            ComboBox editModeSelector = this.FindControl<ComboBox>("EditModeSelector");
            WorldCellPreviewPanel.EditMode editMode = (WorldCellPreviewPanel.EditMode)editModeSelector.SelectedItem;
            WorldCellPreviewPanel worldCellPreviewPanel = this.FindControl<WorldCellPreviewPanel>("WorldCellPreviewPanel");

            if (worldCellPreviewPanel.SelectedTileX == null || worldCellPreviewPanel.SelectedTileY == null) return;

            int tx = worldCellPreviewPanel.SelectedTileX.Value;
            int ty = worldCellPreviewPanel.SelectedTileY.Value;

            if (editMode == WorldCellPreviewPanel.EditMode.DestructableBlock)
            {
                DestructableBlockPlacementDefinition placement = worldMapCell.GetDestructableBlockPlacementAtPosition(tx, ty);
                worldMapCell.DestructableBlockPlacements.Remove(placement);
            }
            else if (editMode == WorldCellPreviewPanel.EditMode.Door)
            {
                DoorPlacementDefinition placement = worldMapCell.GetDoorPlacementAtPosition(tx, ty);
                worldMapCell.DoorPlacements.Remove(placement);
            }
            else if (editMode == WorldCellPreviewPanel.EditMode.Teleporter)
            {
                TeleporterDefinition placement = worldMapCell.GetTeleporterPlacementAtPosition(tx, ty);
                worldMapCell.TeleporterPlacements.Remove(placement);
            }
            else if (editMode == WorldCellPreviewPanel.EditMode.Item)
            {
                ItemPlacementDefinition placement = worldMapCell.GetItemPlacementAtPosition(tx, ty);
                worldMapCell.ItemPlacements.Remove(placement);
            }
            else if (editMode == WorldCellPreviewPanel.EditMode.Saver)
            {
                SaverDefinition placement = worldMapCell.GetSaverPlacementAtPosition(tx, ty);
                worldMapCell.SaverPlacements.Remove(placement);
            }
            else if (editMode == WorldCellPreviewPanel.EditMode.Enemy)
            {
                EnemyPlacementDefinition placement = worldMapCell.GetEnemyPlacementAtPosition(tx, ty);
                worldMapCell.EnemyPlacements.Remove(placement);
            }
            else if (editMode == WorldCellPreviewPanel.EditMode.Victory)
            {
                VictoryDefinition placement = worldMapCell.GetVictoryPlacementAtPosition(tx, ty);
                worldMapCell.VictoryPlacements.Remove(placement);
            }
        }

    }
}
