﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameMakerAvalonia.Forms;
using GameMakerAvalonia.Models;
using GameMakerAvalonia.PreviewPanels;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameMakerAvalonia.Panels
{
    public class WorldsPanel : UserControl
    {
        private Session session;
        private IList<WorldMapDefinition> resources;


        public WorldsPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
            this.FindControl<Button>("WorldAddButton").Click += WorldAddButton_Click;
            this.FindControl<Button>("WorldDeleteButton").Click += WorldDeleteButton_Click;
            this.FindControl<Button>("WorldEditButton").Click += WorldEditButton_Click;

            var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");
            worldPreviewPanel.AllowClicks = false;
        }


        public void SetSession(Session session)
        {
            this.session = session;

            var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");
            worldPreviewPanel.SetSession(session);

            LoadSession();
        }


        private void LoadSession()
        {
            this.resources = this.session.RawContentManager.WorldMapRepository.GetListOfContent();

            var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;
        }


        private void ListGridView_CurrentCellChanged(object sender, EventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem != null)
            {
                WorldMapDefinition worldMap = (WorldMapDefinition)grid.SelectedItem;

                var worldPreviewPanel = this.FindControl<WorldPreviewPanel>("WorldPreviewPanel");
                worldPreviewPanel.SetWorldMap(worldMap);

                worldPreviewPanel.InvalidateVisual();
            }
        }


        private void WorldAddButton_Click(object sender, RoutedEventArgs e)
        {
            WorldMapDefinition worldMap = new WorldMapDefinition();

            this.session.RawContentManager.WorldMapRepository.AddContent(worldMap);

            var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = null;
            LoadSession();
        }


        private async void WorldDeleteButton_Click(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Select a world",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);
                
                return;
            }

            WorldMapDefinition worldMap = (WorldMapDefinition)grid.SelectedItem;

            this.session.RawContentManager.WorldMapRepository.RemoveContent(worldMap.Key);

            grid.Items = null;
            LoadSession();
        }


        private async void WorldEditButton_Click(object sender, RoutedEventArgs e)
        {
            var grid = this.FindControl<DataGrid>("ListGridView");

            if (grid.SelectedItem == null)
            {
                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
                {
                    ContentTitle = "Error",
                    ContentMessage = "Select a world",
                    Style = Style.Windows,
                    Icon = MessageBox.Avalonia.Enums.Icon.Error
                });
                await dialog.ShowDialog((Window)this.VisualRoot);

                return;
            }

            WorldMapDefinition worldMap = (WorldMapDefinition)grid.SelectedItem;

            WorldForm form = new WorldForm();
            form.SetSession(this.session);
            form.SetWorldMap(worldMap);
            await form.ShowDialog((Window)this.VisualRoot);
        }



    }
}
