﻿using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using System;

namespace GameResources.Logic.Physics
{
    public class CollisionDetection
    {
        private readonly IMoveableObject moveableObject;
        private readonly WorldMapCell worldMapCell;
        private readonly GameState gameState;


        public CollisionDetection(IMoveableObject moveableObject, WorldMapCell worldMapCell, GameState gameState)
        {
            this.moveableObject = moveableObject;
            this.worldMapCell = worldMapCell;
            this.gameState = gameState;
        }


        public bool HasCollidedInWorld()
        {
            return HasCollidedInWorldAtPosition(moveableObject.Position.PositionX, moveableObject.Position.PositionY);
        }


        public bool HasCollidedInWorldAtPosition(long px, long py)
        {
            Rectangle objectBounds = moveableObject.GetCollisionAreaAtPosition(px, py);

            int maxX = worldMapCell.GetMaxTileX();
            int maxY = worldMapCell.GetMaxTileY();
            if (px > maxX * TileSet.DRAW_SIZE) return true;
            if (py > maxY * TileSet.DRAW_SIZE) return true;
            if (px < 0) return true;
            if (py < 0) return true;

            int x1 = (int)Math.Floor((objectBounds.X - 1) / (double)TileSet.DRAW_SIZE);
            int y1 = (int)Math.Floor((objectBounds.Y - 1) / (double)TileSet.DRAW_SIZE);
            int x2 = (int)Math.Ceiling((objectBounds.X + objectBounds.Width + 1) / (double)TileSet.DRAW_SIZE);
            int y2 = (int)Math.Ceiling((objectBounds.Y + objectBounds.Height + 1) / (double)TileSet.DRAW_SIZE);

            if (x1 < 0) x1 = 0;
            if (y1 < 0) y1 = 0;
            if (x2 > maxX) x2 = maxX;
            if (y2 > maxY) y2 = maxY;

            for (int x = x1; x < x2; x++)
            {
                for (int y = y1; y < y2; y++)
                {
                    Rectangle tileBounds = new Rectangle(x * TileSet.DRAW_SIZE, y * TileSet.DRAW_SIZE, TileSet.DRAW_SIZE, TileSet.DRAW_SIZE);

                    if (objectBounds.Intersects(tileBounds))
                    {
                        TileSet tileSet1 = worldMapCell.TileSetsLayer1[x][y];
                        if (tileSet1?.Passable == false) return true;

                        TileSet tileSet2 = worldMapCell.TileSetsLayer2[x][y];
                        if (tileSet2?.Passable == false) return true;
                    }
                }
            }


            for(int i = 0; i < this.gameState.ActiveDestructableBlocks.Count; i++)
            {
                DestructableBlock block = this.gameState.ActiveDestructableBlocks[i];

                if (objectBounds.Intersects(block.GetCurrentCollisionArea()))
                {
                    if (block.IsDestroyed == false) return true;
                }
            }

            for(int i = 0; i < this.gameState.ActiveDoors.Count; i++) 
            {
                Door door = this.gameState.ActiveDoors[i];

                if (objectBounds.Intersects(door.GetCurrentCollisionArea()))
                {
                    if (door.IsDestroyed == false) return true;
                }
            }

            for(int i = 0; i < this.gameState.ActiveEnemies.Count; i++)
            {
                Enemy enemy = this.gameState.ActiveEnemies[i];

                if ((enemy.IsExpired == false) && (enemy != moveableObject) & (enemy.IsFrozen))
                {
                    if (objectBounds.Intersects(enemy.GetCurrentCollisionArea()))
                    {
                        return true;
                    }
                }
            }

            return false;
        }


    }
}
