﻿using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.World;
using RawResources.Models;


namespace GameResources.Logic.Physics
{
    public class PlayerTeleporter
    {
        private readonly IMoveableObject moveableObject;
        private readonly GameState gameState;

        private enum Direction { NONE = 0, TOP = 1, BOTTOM = 2, LEFT = 3, RIGHT = 4 }


        public PlayerTeleporter(IMoveableObject moveableObject, GameState gameState)
        {
            this.moveableObject = moveableObject;
            this.gameState = gameState;
        }


        public void TeleportIfPossible()
        {
            WorldPosition position = moveableObject.Position;
            Direction direction = Direction.NONE;
            WorldMapCell cell = gameState.GetActiveWorldMapCell();

            int maxPositionX = cell.GetMaxTileX() * TileSet.DRAW_SIZE;
            int maxPositionY = cell.GetMaxTileY() * TileSet.DRAW_SIZE;
            int toleranceX = 16;
            int toleranceY = 16;

            //detect if player is on edge of screen
            if (position.PositionX <= toleranceX) direction = Direction.LEFT;
            else if (position.PositionY <= toleranceY) direction = Direction.TOP;
            else if (position.PositionX >= maxPositionX - toleranceX) direction = Direction.RIGHT;
            else if (position.PositionY >= maxPositionY - toleranceY) direction = Direction.BOTTOM;

            if (direction != Direction.NONE)
            {
                WorldPosition newPosition = GetNewPosition(cell, position, direction);

                if (newPosition != null)
                {
                    TeleportToPosition(newPosition);
                }
            }
        }


        public void TeleportToPosition(WorldPosition destination)
        {
            moveableObject.Position.WorldMapKey = destination.WorldMapKey;
            moveableObject.Position.WorldCellX = destination.WorldCellX;
            moveableObject.Position.WorldCellY = destination.WorldCellY;
            moveableObject.Position.PositionX = destination.PositionX;
            moveableObject.Position.PositionY = destination.PositionY;

            moveableObject.Motion.Reset();

            gameState.ActivateWorldMapCell();
        }


        private WorldPosition GetNewPosition(WorldMapCell cell, WorldPosition position, Direction direction)
        {
            WorldPosition newposition = null;
            int cellDX = (int)(position.PositionX / (WorldMapCellDefinition.NUM_TILES_WIDE * TileSet.DRAW_SIZE));
            int cellDY = (int)(position.PositionY / (WorldMapCellDefinition.NUM_TILES_HIGH * TileSet.DRAW_SIZE));

            if (direction == Direction.TOP && moveableObject.Motion.VelocityY < 0)
            {
                WorldMapCell newCell = gameState.GetActiveWorldMap().GetCellAtLocation(position.WorldCellX + cellDX, position.WorldCellY + cellDY - 1);
                (int, int) positionDelta = GetPositionAdjustment(cell, newCell, cellDX, cellDY, direction);

                if (newCell != null)
                {
                    newposition = new WorldPosition()
                    {
                        WorldMapKey = position.WorldMapKey,
                        WorldCellX = newCell.WorldMapX,
                        WorldCellY = newCell.WorldMapY,
                        PositionX = position.PositionX + positionDelta.Item1,
                        PositionY = (newCell.GetMaxTileY() * TileSet.DRAW_SIZE) - TileSet.DRAW_SIZE + positionDelta.Item2
                    };
                }
            }
            else if (direction == Direction.BOTTOM && moveableObject.Motion.VelocityY > 0)
            {
                WorldMapCell newCell = gameState.GetActiveWorldMap().GetCellAtLocation(position.WorldCellX + cellDX, position.WorldCellY + cellDY + 1);
                (int, int) positionDelta = GetPositionAdjustment(cell, newCell, cellDX, cellDY, direction);

                if (newCell != null)
                {
                    newposition = new WorldPosition()
                    {
                        WorldMapKey = position.WorldMapKey,
                        WorldCellX = newCell.WorldMapX,
                        WorldCellY = newCell.WorldMapY,
                        PositionX = position.PositionX + positionDelta.Item1,
                        PositionY = (TileSet.DRAW_SIZE*3) + positionDelta.Item2
                    };
                }
            }
            else if (direction == Direction.LEFT && moveableObject.Motion.VelocityX < 0)
            {
                WorldMapCell newCell = gameState.GetActiveWorldMap().GetCellAtLocation(position.WorldCellX + cellDX - 1, position.WorldCellY + cellDY);
                (int, int) positionDelta = GetPositionAdjustment(cell, newCell, cellDX, cellDY, direction);

                if (newCell != null)
                {
                    newposition = new WorldPosition()
                    {
                        WorldMapKey = position.WorldMapKey,
                        WorldCellX = newCell.WorldMapX,
                        WorldCellY = newCell.WorldMapY,
                        PositionX = newCell.GetMaxTileX() * TileSet.DRAW_SIZE + positionDelta.Item1 - (2*TileSet.DRAW_SIZE),
                        PositionY = position.PositionY + positionDelta.Item2
                    };
                }
            }
            else if (direction == Direction.RIGHT && moveableObject.Motion.VelocityX > 0)
            {
                WorldMapCell newCell = gameState.GetActiveWorldMap().GetCellAtLocation(position.WorldCellX + cellDX + 1, position.WorldCellY + cellDY);
                (int, int) positionDelta = GetPositionAdjustment(cell, newCell, cellDX, cellDY, direction);

                if (newCell != null)
                {
                    newposition = new WorldPosition()
                    {
                        WorldMapKey = position.WorldMapKey,
                        WorldCellX = newCell.WorldMapX,
                        WorldCellY = newCell.WorldMapY,
                        PositionX = TileSet.DRAW_SIZE + positionDelta.Item1 + (1 * TileSet.DRAW_SIZE),
                        PositionY = position.PositionY + positionDelta.Item2
                    };
                }
            }

            return newposition;
        }


        private (int, int) GetPositionAdjustment(WorldMapCell currentCell, WorldMapCell newCell, int cellDX, int cellDY, Direction direction)
        {
            int dx = 0;
            int dy = 0;
            int width = WorldMapCellDefinition.NUM_TILES_WIDE * TileSet.DRAW_SIZE;
            int height = WorldMapCellDefinition.NUM_TILES_HIGH * TileSet.DRAW_SIZE;

            if (newCell == null) return (0, 0);

            if (direction == Direction.BOTTOM)
            {
                dx = currentCell.WorldMapX - newCell.WorldMapX;
            }
            else if (direction == Direction.TOP)
            {
                dx = currentCell.WorldMapX - newCell.WorldMapX;
            }
            else if (direction == Direction.LEFT)
            {
                dy = currentCell.WorldMapY - newCell.WorldMapY;
            }
            else if (direction == Direction.RIGHT)
            {
                dy = currentCell.WorldMapY - newCell.WorldMapY;
            }


            dx *= width;
            dy *= height;

            return (dx, dy);
        }


    }
}
