﻿using GameResources.Manager;
using GameResources.Models.Game;
using GameResources.Models.Samus;
using GameResources.Models.Weapons;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using System.Collections.Generic;


namespace GameResources.Logic.Weapons
{
    public class WeaponCollisionDetection
    {
        private IGameContentManager gameContentManager;
        private GameState gameState;
        private IWeaponSprite weaponSprite;
        private IList<Enemy> previouslyHitEnemies;

        public WeaponCollisionDetection(GameState gameState, IGameContentManager gameContentManager, IWeaponSprite weaponSprite, IList<Enemy> previouslyHitEnemies)
        {
            this.gameState = gameState;
            this.gameContentManager = gameContentManager;
            this.weaponSprite = weaponSprite;

            this.previouslyHitEnemies = (previouslyHitEnemies == null) ? new List<Enemy>() : previouslyHitEnemies;
        }


        public bool DealWithCollision()
        {
            bool hit = false;
            Rectangle objectBounds = weaponSprite.GetCurrentCollisionArea();

            foreach(DestructableBlock block in gameState.ActiveDestructableBlocks)
            {
                if(block.IsDestroyed == false)
                {
                    if(block.GetCurrentCollisionArea().Intersects(objectBounds))
                    {
                        hit = true;

                        block.AttemptDestruction(weaponSprite.DamageType, gameState, gameContentManager);
                    }
                }
            }

            foreach(Door door in gameState.ActiveDoors)
            {
                if(door.IsDestroyed == false)
                {
                    if (door.GetCurrentCollisionArea().Intersects(objectBounds))
                    {
                        hit = true;

                        door.AttemptDestruction(weaponSprite.DamageType, gameState, gameContentManager);
                    }
                }
            }

            foreach(Enemy enemy in gameState.ActiveEnemies)
            {
                if(previouslyHitEnemies.Contains(enemy) == false)
                {
                    if(enemy.GetCurrentCollisionArea().Intersects(objectBounds))
                    {
                        hit = true;

                        enemy.AttemptDamage(weaponSprite.DamageType, weaponSprite.WeaponModifier, weaponSprite.Damage, gameState, gameContentManager);
                        previouslyHitEnemies.Add(enemy);
                    }
                }
            }

            if (weaponSprite.DamageType == RawResources.Models.DamageType.Bomb)
            {
                if (gameState.Player.MotionState.IsBall() && gameState.Player.GetCurrentCollisionArea().Intersects(objectBounds))
                {
                    BombJump();
                }
            }

            return hit;
        }



        private void BombJump()
        {
            Player player = gameState.Player;
            int heightVariance = 8;

            if (weaponSprite.Position.PositionY >= (player.Position.PositionY-heightVariance))
            {
                player.Motion.VelocityY += -12;

                if (player.MotionState.IsFacingLeft()) player.MotionState.Current = PlayerMotionState.State.Ball_Fall_Left;
                else player.MotionState.Current = PlayerMotionState.State.Ball_Fall_Right;
            }          
        }

    }
}
