﻿using GameResources.Logic.Game;
using GameResources.Logic.Physics;
using GameResources.Logic.Weapons;
using GameResources.Manager;
using GameResources.Models.Characters;
using GameResources.Models.Effects;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.Weapons;
using GameResources.Models.World;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System;


namespace GameResources.Models.Samus
{
    public class Player : IBattleObject
    {
        public Motion Motion { get; set; }
        public WorldPosition Position { get; set; }
        public CharacterStats CharacterStats { get; set; }
        public PlayerCharacterSet CharacterSet { get; set; }
        public PlayerMotionState MotionState { get; set; }
        public PlayerAttackState AttackState { get; set; }
        public PlayerAimState AimState { get; set; }
        public PlayerInventory Inventory { get; set; }

        private long lastAnimationUpdate;
        private long lastMoveUpdate;
        private long lastWeaponChangeUpdate;
        public InputMap InputMap { get; set; }
        private bool isSpeedBoosting;

        private bool IsDamaged;
        private long lastDamageUpdate;
        private long lastEnemyCollisionUpdate;


        public Player()
        {
            this.Motion = new Motion();
            this.MotionState = new PlayerMotionState();
            this.AttackState = new PlayerAttackState();
            this.AimState = new PlayerAimState();
            this.CharacterStats = new CharacterStats();
            this.Inventory = new PlayerInventory();
            this.IsDamaged = false;

            this.lastAnimationUpdate = 0;
            this.lastMoveUpdate = 0;
            this.lastWeaponChangeUpdate = 0;
            this.lastDamageUpdate = 0;
            this.lastEnemyCollisionUpdate = 0;
        }


        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            Rectangle screenDrawRect = new Rectangle((int)(Position.PositionX - dx - (CharacterSet.GetCurrentTextureWidth() / 2)), (int)(Position.PositionY - dy - CharacterSet.GetCurrentTextureHeight()), CharacterSet.GetCurrentTextureWidth(), CharacterSet.GetCurrentTextureHeight());

            Color color = Color.White;

            if(IsDamaged)
            {
                color = Color.Red;
            }
            else if (this.Inventory.HasVariaSuit)
            {
                color = new Color(225, 139, 0); //orange
            }
            else if (this.Inventory.HasGravitySuit)
            {
                color = new Color(180, 130, 255); //purple
            }

            if (this.AttackState.Current == PlayerAttackState.State.Charged || this.AttackState.Current == PlayerAttackState.State.Charging)
            {
                double interval = (this.AttackState.Current == PlayerAttackState.State.Charging) ? 3 : 2;

                if (this.AttackState.ChargingTime >= 0.2)
                {
                    if ((Math.Round(this.AttackState.ChargingTime, 1) * 10) % interval <= 0.01) color = Color.Red;
                }
            }
            
            if(this.isSpeedBoosting)
            {
                int sbX = this.MotionState.IsFacingLeft() ? 1 : -1;

                spriteBatch.Draw(CharacterSet.GetCurrentTexture(), new Rectangle(screenDrawRect.X + (sbX * 48), screenDrawRect.Y, screenDrawRect.Width, screenDrawRect.Height), Color.Blue);
                spriteBatch.Draw(CharacterSet.GetCurrentTexture(), new Rectangle(screenDrawRect.X + (sbX * 32), screenDrawRect.Y, screenDrawRect.Width, screenDrawRect.Height), Color.Purple);
                spriteBatch.Draw(CharacterSet.GetCurrentTexture(), new Rectangle(screenDrawRect.X + (sbX * 16), screenDrawRect.Y, screenDrawRect.Width, screenDrawRect.Height), Color.Red);
            }

            spriteBatch.Draw(CharacterSet.GetCurrentTexture(), screenDrawRect, color);
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            bool refreshAnimation = false;
            bool toMove = false;

            UpdateFromInput(ticks, gameContentManager, gameState);

            lastAnimationUpdate += ticks;
            lastMoveUpdate += ticks;
            lastWeaponChangeUpdate += ticks;
            lastDamageUpdate += ticks;
            lastEnemyCollisionUpdate += ticks;

            if (lastWeaponChangeUpdate >= (TimeSpan.TicksPerSecond * 0.2))
            {
                if (InputMap.ChangeItem)
                {
                    this.Inventory.ChangeSelectedWeapon();

                    lastWeaponChangeUpdate = 0;
                }
            }

            if (IsDamaged && (lastDamageUpdate >= (TimeSpan.TicksPerSecond * 0.1)))
            {
                IsDamaged = false;
            }

            if (lastAnimationUpdate >= (TimeSpan.TicksPerSecond * 0.06))
            {
                refreshAnimation = true;
                lastAnimationUpdate = 0;
            }

            if (lastMoveUpdate >= (TimeSpan.TicksPerSecond * 0.0085))
            {
                toMove = true;
                lastMoveUpdate = 0;
            }

            if (refreshAnimation)
            {
                this.CharacterSet.UpdateFrame(this.MotionState, this.AimState, this.AttackState, this.Inventory);
            }

            UpdateSpeedBoosting(gameState, ticks);
            SpeedBoostDestroy(gameState, gameContentManager, ticks);
            ScrewAttackDestroy(gameState, gameContentManager, ticks);

            if (lastEnemyCollisionUpdate >= (TimeSpan.TicksPerSecond * 0.5))
            {
                CheckForCollisionWithEnemy(gameState, gameContentManager, ticks);
            }

            Shoot(gameState, gameContentManager);

            if (toMove && (IsDamaged == false))
            {
                MovePlayer(gameState, ticks);

                CheckForCellTransport(gameState);
            }
        }


        private void CheckForCollisionWithEnemy(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            Rectangle bounds = GetCollisionAreaAtPosition((int)(this.Position.PositionX + this.Motion.VelocityX), (int)(this.Position.PositionY + this.Motion.VelocityY));

            for(int i = 0; i < gameState.ActiveEnemies.Count; i++)
            {
                Enemy enemy = gameState.ActiveEnemies[i];

                if ((enemy.IsExpired == false) && (enemy.IsFrozen == false))
                {
                    if (bounds.Intersects(enemy.GetCurrentCollisionArea()))
                    {
                        this.CharacterStats.Health -= enemy.Attack;
                        IsDamaged = true;

                        lastDamageUpdate = 0;
                        lastEnemyCollisionUpdate = 0;

                        gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Samus Hit").Play();

                        //bounce back
                        float vx = this.Motion.VelocityX;
                        float vy = this.Motion.VelocityY;

                        this.Motion.VelocityX = 0;
                        this.Motion.VelocityY = -40;
                        PhysicsEngine physics = new PhysicsEngine(this, gameState.GetActiveWorldMapCell(), gameState, false);
                        physics.Update();

                        this.Motion.VelocityX = vx;
                        this.Motion.VelocityY = vy;


                        //explode
                        if (enemy.IsExplosive)
                        {
                            enemy.IsExpired = true;

                            SmallEnemyDeathEffect deathEffect = new SmallEnemyDeathEffect(gameContentManager.TextureRepository, gameContentManager.SoundEffectRepository, this.Position.PositionX, this.Position.PositionY - enemy.GetHeight() / 2);
                            gameState.ActiveEffects.Add(deathEffect);
                        }

                        return;
                    }
                }
            }
        }


        private void ScrewAttackDestroy(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            bool isScrewAttacking = this.MotionState.IsScrewAttacking(this.Inventory);

            if(isScrewAttacking)
            {
                Rectangle bounds = GetCollisionAreaAtPosition((int)(this.Position.PositionX + this.Motion.VelocityX), (int)(this.Position.PositionY + this.Motion.VelocityY));

                for(int i = 0; i < gameState.ActiveDestructableBlocks.Count; i++)
                {
                    DestructableBlock block = gameState.ActiveDestructableBlocks[i];

                    if (bounds.Intersects(block.GetCurrentCollisionArea()))
                    {
                        block.AttemptDestruction(DamageType.ScrewAttack, gameState, gameContentManager);
                    }
                }

                for (int i = 0; i < gameState.ActiveEnemies.Count; i++)
                {
                    Enemy enemy = gameState.ActiveEnemies[i];

                    if ((enemy.IsScrewable) && (bounds.Intersects(enemy.GetCurrentCollisionArea())))
                    {
                        enemy.AttemptDamage(DamageType.ScrewAttack, new WeaponModifier(), 1000, gameState, gameContentManager);
                    }
                }
            }
        }


        private void UpdateSpeedBoosting(GameState gameState, long ticks)
        {
            if (this.Inventory.HasSpeedBoost == false)
            {
                isSpeedBoosting = false;
                return;
            }
                
            if(gameState.IsInteractingWithWater())
            {
                if (this.Inventory.HasGravitySuit) isSpeedBoosting = this.MotionState.IsSpeedBoosting();
                else isSpeedBoosting = false;
            }
            else
            {
                isSpeedBoosting = this.MotionState.IsSpeedBoosting();
            }
        }


        private void SpeedBoostDestroy(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            if(isSpeedBoosting)
            {
                Rectangle bounds = GetCollisionAreaAtPosition((int)(this.Position.PositionX + this.Motion.VelocityX), (int)(this.Position.PositionY + this.Motion.VelocityY));

                for (int i = 0; i < gameState.ActiveDestructableBlocks.Count; i++)
                {
                    DestructableBlock block = gameState.ActiveDestructableBlocks[i];

                    if (bounds.Intersects(block.GetCurrentCollisionArea()))
                    {
                        block.AttemptDestruction(DamageType.SpeedBoost, gameState, gameContentManager);
                    }
                }
            }
        }


        public void Shoot(GameState gameState, IGameContentManager gameContentManager)
        {
            if (this.AttackState.Current == PlayerAttackState.State.Shoot || this.AttackState.Current == PlayerAttackState.State.ShootCharged)
            {
                Point aimDirection = this.AimState.GetAimDirection(this.MotionState);
                WorldPosition weaponSpawnPoint = GetWeaponPoint();

                this.Inventory.CheckForNoAmmo();

                if (this.MotionState.IsBall() == false)
                {
                    if (this.Inventory.SelectedWeapon == PlayerInventory.Weapons.Missile)
                    {
                        if (this.Inventory.Missiles > 0)
                        {
                            this.Inventory.Missiles--;
                            MissileShooter missileShooter = new MissileShooter(gameState, gameContentManager);
                            missileShooter.Do(this, weaponSpawnPoint, aimDirection);
                        }
                    }
                    else if (this.Inventory.SelectedWeapon == PlayerInventory.Weapons.SuperMissle)
                    {
                        if (this.Inventory.SuperMissiles > 0)
                        {
                            this.Inventory.SuperMissiles--;
                            SuperMissileShooter superMissileShooter = new SuperMissileShooter(gameState, gameContentManager);
                            superMissileShooter.Do(this, weaponSpawnPoint, aimDirection);
                        }
                    }
                    else
                    {
                        WeaponModifier weaponModifier = new WeaponModifier()
                        {
                            IsPlasma = this.Inventory.HasPlasmaBeam,
                            IsWave = this.Inventory.HasWaveBeam,
                            IsIce = this.Inventory.HasIceBeam,
                            IsSpazer = this.Inventory.HasSpazerBeam,
                            IsCharged = (this.AttackState.Current == PlayerAttackState.State.ShootCharged)
                        };

                        BasicShooter basicShooter = new BasicShooter(gameState, gameContentManager);
                        basicShooter.Do(this, weaponSpawnPoint, aimDirection, weaponModifier);
                    }
                }
                else
                {
                    if (this.Inventory.SelectedWeapon == PlayerInventory.Weapons.SuperBomb)
                    {
                        if(this.Inventory.SuperBombs > 0)
                        {
                            this.Inventory.SuperBombs--;
                            SuperBombPlacer bombPlacer = new SuperBombPlacer(gameState, gameContentManager);
                            bombPlacer.Do(this, weaponSpawnPoint);
                        }
                    }
                    else
                    {
                        if (this.Inventory.HasBombs)
                        {
                            BombPlacer bombPlacer = new BombPlacer(gameState, gameContentManager);
                            bombPlacer.Do(this, weaponSpawnPoint);
                        }
                    }
                }
            }
        }


        public void UpdateFromInput(long ticks, IGameContentManager gameContentManager, GameState gameState)
        {
            bool wasBall = this.MotionState.IsBall();

            this.MotionState.Update(InputMap, ticks, this.Motion, this.Inventory, gameState);
            this.AttackState.Update(InputMap, ticks, this.Inventory, this.MotionState, gameContentManager.SoundEffectRepository);
            this.AimState.Update(InputMap, ticks, this.MotionState);

            bool isBall = this.MotionState.IsBall();

            if(isBall == false && wasBall)
            {
                PlayerPositionAdjuster positionAdjuster = new PlayerPositionAdjuster(this, gameState);
                positionAdjuster.AdjustIfNeeded();
            }

            Teleporter teleporter = gameState.IsInteractingWithATeleporter();
            if(teleporter != null && InputMap.Up)
            {
                PlayerTeleporter playerTeleporter = new PlayerTeleporter(this, gameState);
                playerTeleporter.TeleportToPosition(teleporter.Destination);
            }

            Saver saver = gameState.IsInteractingWithASaver();
            if(saver != null && InputMap.Up)
            {
                this.Inventory.Missiles = this.Inventory.MissilesMax;
                this.Inventory.SuperMissiles = this.Inventory.SuperMissilesMax;
                this.Inventory.SuperBombs = this.Inventory.SuperBombsMax;
                this.CharacterStats.Health = this.CharacterStats.MaximumHealth;

                GameSaver gameSaver = new GameSaver();
                gameSaver.Save(gameState);

                TextDisplayEffect displayEffect = new TextDisplayEffect(gameContentManager, this.Position, gameState.GetActiveWorldMapCell(), "Game Saved Successfully!");
                gameState.ActiveEffects.Add(displayEffect);
            }

            Victory victory = gameState.IsInteractingWithAVictory();
            if(victory != null && InputMap.Up)
            {
                gameState.IsVictory = true;
            }
        }


        private void CheckForCellTransport(GameState gameState)
        {
            PlayerTeleporter playerTeleporter = new PlayerTeleporter(this, gameState);
            playerTeleporter.TeleportIfPossible();
        }


        private void MovePlayer(GameState gameState, long ticks)
        {
            double dt = (double)ticks / TimeSpan.TicksPerSecond;
            bool isInWater = gameState.IsInteractingWithWater() && (Inventory.HasGravitySuit == false);
            int movementSpeed = isSpeedBoosting ? 15 : isInWater ? 2 : 5;
            int jumpSpeed = isSpeedBoosting ? 18 : isInWater ? 9 : 12;

            //this.Motion.VelocityX = 0;
            //this.Motion.VelocityY = 0;

            if (this.MotionState.Current == PlayerMotionState.State.Running_Left) this.Motion.VelocityX = -movementSpeed;
            else if (this.MotionState.Current == PlayerMotionState.State.Running_Right) this.Motion.VelocityX = movementSpeed;
            else if (this.MotionState.Current == PlayerMotionState.State.Stand_Left) this.Motion.VelocityX = 0;
            else if (this.MotionState.Current == PlayerMotionState.State.Stand_Right) this.Motion.VelocityX = 0;
            else if (this.MotionState.Current == PlayerMotionState.State.Crouch_Left) this.Motion.VelocityX = 0;
            else if (this.MotionState.Current == PlayerMotionState.State.Crouch_Right) this.Motion.VelocityX = 0;
            else if (this.MotionState.Current == PlayerMotionState.State.Jump_Left || 
                     this.MotionState.Current == PlayerMotionState.State.Jump_Right ||
                     this.MotionState.Current == PlayerMotionState.State.Ball_Jump_Left ||
                     this.MotionState.Current == PlayerMotionState.State.Ball_Jump_Right)
            {
                this.Motion.VelocityY = -jumpSpeed;

                if (this.InputMap.Left) this.Motion.VelocityX = -movementSpeed;
                else if (this.InputMap.Right) this.Motion.VelocityX = movementSpeed;
                else this.Motion.VelocityX = 0;
            }
            else if (this.MotionState.Current == PlayerMotionState.State.Fall_Left || 
                     this.MotionState.Current == PlayerMotionState.State.Fall_Right ||
                     this.MotionState.Current == PlayerMotionState.State.Ball_Fall_Left ||
                     this.MotionState.Current == PlayerMotionState.State.Ball_Fall_Right)
            {
                if (this.InputMap.Left) this.Motion.VelocityX = -movementSpeed;
                else if (this.InputMap.Right) this.Motion.VelocityX = movementSpeed;
                else this.Motion.VelocityX = 0;
            }
            else if (this.MotionState.Current == PlayerMotionState.State.Ball_Rolling_Left) this.Motion.VelocityX = -movementSpeed;
            else if (this.MotionState.Current == PlayerMotionState.State.Ball_Rolling_Right) this.Motion.VelocityX = movementSpeed;
            else if (this.MotionState.Current == PlayerMotionState.State.Ball_Left) this.Motion.VelocityX = 0;
            else if (this.MotionState.Current == PlayerMotionState.State.Ball_Right) this.Motion.VelocityX = 0;

            ApplyGravity(ticks, dt);
            

            WorldMapCell worldMapCell = gameState.GetActiveWorldMapCell();
            PhysicsEngine physics = new PhysicsEngine(this, worldMapCell, gameState, false);
            physics.Update();
        }


        private void ApplyGravity(long ticks, double dt2)
        {
            this.Motion.VelocityY += 1;
            if (this.Motion.VelocityY > 10) this.Motion.VelocityY = 10;
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(Position.PositionX, Position.PositionY);
        }


        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            int w = 32;
            int h = 64;

            
            if (this.MotionState.IsBall())
            {
                w = 32;
                h = 32;
            }
            else if (this.MotionState.IsJumping() || this.MotionState.IsCrouching())
            {
                w = 32;
                h = 64;
            }

            Rectangle area = new Rectangle((int)(x - (w / 2)), (int)(y - h), w, h);

            return area;
        }


        private WorldPosition GetWeaponPoint()
        {
            WorldPosition position = this.Position.Copy();
            Point delta = this.AimState.GetWeaponPoint(this.MotionState);

            position.PositionX += delta.X;
            position.PositionY += delta.Y;

            return position;
        }


        public void AddItem(Item item, IGameContentManager gameContentManager, GameState gameState)
        {
            if(item.ItemType == ItemType.PickupHealthSmall || 
               item.ItemType == ItemType.PickupHealthLarge || 
               item.ItemType == ItemType.PickupMissiles ||
               item.ItemType == ItemType.PickupSuperMissiles ||
               item.ItemType == ItemType.PickupSuperBombs)
            {
                if (item.ItemType == ItemType.PickupHealthLarge) this.CharacterStats.Health += item.GetQuantity();
                else if (item.ItemType == ItemType.PickupHealthSmall) this.CharacterStats.Health += item.GetQuantity();
                else if (item.ItemType == ItemType.PickupMissiles) this.Inventory.Missiles += item.GetQuantity();
                else if (item.ItemType == ItemType.PickupSuperMissiles) this.Inventory.SuperMissiles += item.GetQuantity();
                else if (item.ItemType == ItemType.PickupSuperBombs) this.Inventory.SuperBombs += item.GetQuantity();

                if (this.CharacterStats.Health > this.CharacterStats.MaximumHealth) this.CharacterStats.Health = this.CharacterStats.MaximumHealth;
                if (this.Inventory.Missiles > this.Inventory.MissilesMax) this.Inventory.Missiles = this.Inventory.MissilesMax;
                if (this.Inventory.SuperMissiles > this.Inventory.SuperMissilesMax) this.Inventory.SuperMissiles = this.Inventory.SuperMissilesMax;
                if (this.Inventory.SuperBombs > this.Inventory.SuperBombsMax) this.Inventory.SuperBombs = this.Inventory.SuperBombsMax;

                gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Pickup").Play();
            }
            else
            {
                if (item.ItemType == ItemType.Bombs) this.Inventory.HasBombs = true;
                else if (item.ItemType == ItemType.ChargeBeam) this.Inventory.HasChargeBeam = true;
                else if (item.ItemType == ItemType.IceBeam) this.Inventory.HasIceBeam = true;
                else if (item.ItemType == ItemType.MorphBall) this.Inventory.HasMorphBall = true;
                else if (item.ItemType == ItemType.PlasmaBeam) this.Inventory.HasPlasmaBeam = true;
                else if (item.ItemType == ItemType.SpazerBeam) this.Inventory.HasSpazerBeam = true;
                else if (item.ItemType == ItemType.SpringBall) this.Inventory.HasSpringBall = true;
                else if (item.ItemType == ItemType.SuperJump) this.Inventory.HasSuperJump = true;
                else if (item.ItemType == ItemType.VariaSuit) this.Inventory.HasVariaSuit = true;
                else if (item.ItemType == ItemType.WaveBeam) this.Inventory.HasWaveBeam = true;
                else if (item.ItemType == ItemType.GravitySuit) this.Inventory.HasGravitySuit = true;
                else if (item.ItemType == ItemType.SpeedBoost) this.Inventory.HasSpeedBoost = true;
                else if (item.ItemType == ItemType.ScrewAttack) this.Inventory.HasScrewAttack = true;
                else if (item.ItemType == ItemType.SpaceJump) this.Inventory.HasSpaceJump = true;
                else if (item.ItemType == ItemType.Missiles)
                {
                    this.Inventory.MissilesMax += item.GetQuantity();
                    this.Inventory.Missiles += item.GetQuantity();
                }
                else if (item.ItemType == ItemType.SuperMissiles)
                {
                    this.Inventory.SuperMissilesMax += item.GetQuantity();
                    this.Inventory.SuperMissiles += item.GetQuantity();
                }
                else if (item.ItemType == ItemType.SuperBombs)
                {
                    this.Inventory.SuperBombsMax += item.GetQuantity();
                    this.Inventory.SuperBombs += item.GetQuantity();
                }
                else if (item.ItemType == ItemType.EnergyTank)
                {
                    this.CharacterStats.MaximumHealth += item.GetQuantity();
                    this.CharacterStats.Health = this.CharacterStats.MaximumHealth;
                }

                ItemPickupEffect effect = new ItemPickupEffect(gameContentManager, item, this.Position, gameState.GetActiveWorldMapCell());
                gameState.ActiveEffects.Add(effect);

                gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Item Pickup").Play();
            }
        }


    }
}
