﻿using GameResources.Logic.Physics;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.World;
using System;


namespace GameResources.Models.Samus
{
    public class PlayerMotionState { 
        public enum State {
            Stand_Left,
            Running_Left,
            Jump_Left,
            Fall_Left,
            Crouch_Left,
            Stand_Right,
            Running_Right,
            Jump_Right,
            Fall_Right,
            Crouch_Right,
            Ball_Left,
            Ball_Right,
            Ball_Rolling_Left,
            Ball_Rolling_Right,
            Ball_Jump_Left,
            Ball_Jump_Right,
            Ball_Fall_Left,
            Ball_Fall_Right
        }


        public State Current { get; set; }
        private long jumpStart;
        private bool canChangeFromBall;
        private long runStart;


        public PlayerMotionState()
        {
            this.Current = State.Stand_Left;

            canChangeFromBall = false;

            jumpStart = 0;
            runStart = 0;
        }


        public void Update(InputMap inputMap, long ticks, Motion motion, PlayerInventory inventory, GameState gameState)
        {
            double maxJumpTime = inventory.HasSuperJump ? 0.25 : 0.125;
            double maxBallJumptime = 0.2;
            double jumpTime = (double)jumpStart / (double)TimeSpan.TicksPerSecond;

            if(IsJumping()) jumpStart += ticks;

            if (IsRunning()) runStart += ticks;
            //else runStart = 0;

            if (Current == State.Stand_Left)
            {
                canChangeFromBall = false;

                if (inputMap.Left)
                {
                    Current = State.Running_Left;
                    runStart = 0;
                }
                else if (inputMap.Right) 
                { 
                    Current = State.Running_Right;
                    runStart = 0;
                }
                else if (inputMap.Down) Current = State.Crouch_Left;
                else if (inputMap.Jump && motion.IsOnGround)
                {
                    Current = State.Jump_Left;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Stand_Right)
            {
                canChangeFromBall = false;

                if (inputMap.Left)
                {
                    Current = State.Running_Left;
                    runStart = 0;
                }
                else if (inputMap.Right)
                {
                    Current = State.Running_Right;
                    runStart = 0;
                }
                else if (inputMap.Down) Current = State.Crouch_Right;
                else if (inputMap.Jump && motion.IsOnGround)
                {
                    Current = State.Jump_Right;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Running_Left)
            {
                if (inputMap.Left == false)
                {
                    Current = State.Stand_Left;
                }
                else if (inputMap.Jump && motion.IsOnGround)
                {
                    Current = State.Jump_Left;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Running_Right)
            {
                if (inputMap.Right == false)
                {
                    Current = State.Stand_Right;
                }
                else if (inputMap.Jump && motion.IsOnGround)
                {
                    Current = State.Jump_Right;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Jump_Left)
            {
                if ((inputMap.Jump == false) || (jumpTime > maxJumpTime))
                {
                    Current = State.Fall_Left;
                }
            }
            else if (Current == State.Jump_Right)
            {
                if ((inputMap.Jump == false) || (jumpTime > maxJumpTime))
                {
                    Current = State.Fall_Right;
                }
            }
            else if (Current == State.Fall_Left)
            {
                runStart = 0;

                if (motion.IsOnGround) Current = State.Stand_Left;
                else if (inputMap.Jump && inventory.HasSpaceJump)
                {
                    if (inputMap.Left) Current = State.Jump_Left;
                    else if (inputMap.Right) Current = State.Jump_Right;
                    else Current = State.Jump_Left;

                    jumpStart = 0;
                }
                else if (inputMap.Right) Current = State.Fall_Right;
            }
            else if (Current == State.Fall_Right)
            {
                runStart = 0;

                if (motion.IsOnGround) Current = State.Stand_Right;
                else if (inputMap.Jump && inventory.HasSpaceJump)
                {
                    if (inputMap.Left) Current = State.Jump_Left;
                    else if (inputMap.Right) Current = State.Jump_Right;
                    else Current = State.Jump_Right;

                    jumpStart = 0;
                }
                else if (inputMap.Left) Current = State.Fall_Left;
            }
            else if (Current == State.Crouch_Left)
            {
                if (inputMap.Up) Current = State.Stand_Left;
                else if (inputMap.Left) Current = State.Stand_Left;
                else if (inputMap.Right) Current = State.Stand_Right;
                else if (inputMap.Down && canChangeFromBall && inventory.HasMorphBall)
                {
                    canChangeFromBall = false;
                    Current = State.Ball_Left;
                }
                else if (inputMap.Down == false) canChangeFromBall = true;
                else if (inputMap.Jump && motion.IsOnGround)
                {
                    Current = State.Jump_Left;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Crouch_Right)
            {
                if (inputMap.Up) Current = State.Stand_Right;
                else if (inputMap.Left) Current = State.Stand_Left;
                else if (inputMap.Right) Current = State.Stand_Right;
                else if (inputMap.Down && canChangeFromBall && inventory.HasMorphBall)
                {
                    canChangeFromBall = false;
                    Current = State.Ball_Right;
                }
                else if (inputMap.Down == false)
                {
                    canChangeFromBall = true;
                }
                else if (inputMap.Jump && motion.IsOnGround)
                {
                    Current = State.Jump_Right;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Ball_Left)
            {
                runStart = 0;

                if (inputMap.Up && IsRoomToStand(gameState)) Current = State.Stand_Left;
                else if (inputMap.Left) Current = State.Ball_Rolling_Left;
                else if (inputMap.Right) Current = State.Ball_Rolling_Right;
                else if (inputMap.Jump && motion.IsOnGround && inventory.HasSpringBall)
                {
                    Current = State.Ball_Jump_Left;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Ball_Right)
            {
                runStart = 0;

                if (inputMap.Up && IsRoomToStand(gameState)) Current = State.Stand_Right;
                else if (inputMap.Left) Current = State.Ball_Rolling_Left;
                else if (inputMap.Right) Current = State.Ball_Rolling_Right;
                else if (inputMap.Jump && motion.IsOnGround && inventory.HasSpringBall)
                {
                    Current = State.Ball_Jump_Right;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Ball_Rolling_Left)
            {
                runStart = 0;

                if (inputMap.Left == false) Current = State.Ball_Left;
                else if (inputMap.Jump && motion.IsOnGround && inventory.HasSpringBall)
                {
                    Current = State.Ball_Jump_Left;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Ball_Rolling_Right)
            {
                runStart = 0;

                if (inputMap.Right == false) Current = State.Ball_Right;
                else if (inputMap.Jump && motion.IsOnGround && inventory.HasSpringBall)
                {
                    Current = State.Ball_Jump_Right;
                    jumpStart = 0;
                }
            }
            else if (Current == State.Ball_Jump_Left)
            {
                runStart = 0;

                if ((inputMap.Jump == false) || (jumpTime > maxBallJumptime))
                {
                    Current = State.Ball_Fall_Left;
                }
            }
            else if (Current == State.Ball_Jump_Right)
            {
                runStart = 0;

                if ((inputMap.Jump == false) || (jumpTime > maxBallJumptime))
                {
                    Current = State.Ball_Fall_Right;
                }
            }
            else if (Current == State.Ball_Fall_Left)
            {
                runStart = 0;

                if (motion.IsOnGround) Current = State.Ball_Left;
            }
            else if (Current == State.Ball_Fall_Right)
            {
                runStart = 0;

                if (motion.IsOnGround) Current = State.Ball_Right;
            }
        }


        private bool IsRoomToStand(GameState gameState)
        {
            WorldMapCell activeCell = gameState.GetActiveWorldMapCell();
            Player player = gameState.Player;

            CollisionDetection detector = new CollisionDetection(player, activeCell, gameState);
            return !detector.HasCollidedInWorldAtPosition(player.Position.PositionX, player.Position.PositionY - 16);
        }


        public bool IsScrewAttacking(PlayerInventory inventory)
        {
            return (IsJumping() && inventory.HasScrewAttack && (IsBall() == false));
        }


        public bool IsSpeedBoosting()
        {
            double speedBoost = 3;

            if (runStart == 0) return false;

            bool correctStates = IsRunning() || IsJumping();

            double runTime = (double)runStart / TimeSpan.TicksPerSecond;

            return (correctStates && (runTime >= speedBoost));
        }


        public bool IsBall()
        {
            return (this.Current == State.Ball_Left || 
                    this.Current == State.Ball_Right ||
                    this.Current == State.Ball_Rolling_Left ||
                    this.Current == State.Ball_Rolling_Right ||
                    this.Current == State.Ball_Fall_Left ||
                    this.Current == State.Ball_Fall_Right ||
                    this.Current == State.Ball_Jump_Left ||
                    this.Current == State.Ball_Jump_Right
                    );
        }


        public bool IsCrouching()
        {
            return (this.Current == State.Crouch_Left || this.Current == State.Crouch_Right);
        }


        public bool IsJumping()
        {
            return (this.Current == State.Jump_Left || 
                    this.Current == State.Jump_Right || 
                    this.Current == State.Fall_Left || 
                    this.Current == State.Fall_Right ||
                    this.Current == State.Ball_Jump_Left ||
                    this.Current == State.Ball_Jump_Right ||
                    this.Current == State.Ball_Fall_Left ||
                    this.Current == State.Ball_Fall_Right
                    );
        }


        public bool IsFacingLeft()
        {
            return 
                (
                    this.Current == State.Stand_Left ||
                    this.Current == State.Running_Left ||
                    this.Current == State.Jump_Left ||
                    this.Current == State.Fall_Left ||
                    this.Current == State.Crouch_Left 
                );
        }


        public bool IsFacingRight()
        {
            return
                (
                    this.Current == State.Stand_Right ||
                    this.Current == State.Running_Right ||
                    this.Current == State.Jump_Right ||
                    this.Current == State.Fall_Right ||
                    this.Current == State.Crouch_Right 
                );
        }


        public bool IsRunning()
        {
            return
                (
                    this.Current == State.Running_Left ||
                    this.Current == State.Running_Right
                );
        }


        


    }
}
