﻿using GameResources.Manager;
using GameResources.Models.Effects;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System;


namespace GameResources.Models.World
{
    public class DestructableBlock : IMoveableObject
    {
        public int Key { get; set; }
        public string Name { get; set; }
        public TileSet TileSet { get; set; }
        public TileSet RevealTileSet { get; set; }
        public DamageType DamageType { get; set; }
        public string OnlyShowIfVariableIsFalse { get; set; }
        public string SetVariableToTrueUponDestruction { get; set; }
        public WorldPosition Position { get; set; }
        public Motion Motion { get; set; }
        public bool IsDestroyed { get; set; }
        public bool IsRevealed { get; set; }
        private long destructionStart;
        private double maxDestructionTime;


        public DestructableBlock()
        {
            this.Motion = new Motion();
            this.Position = new WorldPosition();
            this.IsDestroyed = false;

            this.maxDestructionTime = 10;

            this.destructionStart = 0;
        }


        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            if (IsDestroyed == false)
            {
                Rectangle rect = new Rectangle((int)(this.Position.PositionX - dx), (int)(this.Position.PositionY - dy), TileSet.DRAW_SIZE, TileSet.DRAW_SIZE);

                Texture2D texture = this.IsRevealed ? this.RevealTileSet.Texture : this.TileSet.Texture;

                spriteBatch.Draw(texture, rect, Color.White);
            }
        }


        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            return new Rectangle((int)this.Position.PositionX, (int)this.Position.PositionY, TileSet.DRAW_SIZE, TileSet.DRAW_SIZE);
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(this.Position.PositionX, this.Position.PositionY);
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            if (this.OnlyShowIfVariableIsFalse.Equals("") == false)
            {
                if (gameState.VariableState.GetVariable(this.OnlyShowIfVariableIsFalse) == true)
                {
                    this.IsDestroyed = true;
                    destructionStart = 0;
                }
            }
            else
            {
                if (this.IsDestroyed)
                {
                    destructionStart += ticks;

                    if (destructionStart >= (TimeSpan.TicksPerSecond * maxDestructionTime))
                    {
                        IsDestroyed = false;
                        destructionStart = 0;
                    }
                }
            }
        }


        public void AttemptDestruction(DamageType damageType, GameState gameState, IGameContentManager gameContentManager)
        {
            if (this.IsDestroyed) return;
            if (this.OnlyShowIfVariableIsFalse.Equals("") == false) return;

            //reveal test
            if((damageType == DamageType.Normal) || (damageType == DamageType.Bomb) || (damageType == DamageType.SuperBomb))
            {
                this.IsRevealed = true;
            }
            
            //destruction test
            if(this.DamageType == DamageType.Missile)
            {
                if(damageType == DamageType.Missile || damageType == DamageType.SuperMissile)
                {
                    this.IsDestroyed = true;
                }
            }
            else if (this.DamageType == DamageType.Bomb)
            {
                if (damageType == DamageType.Bomb || damageType == DamageType.SuperBomb)
                {
                    this.IsDestroyed = true;
                }
            }
            else if (this.DamageType == damageType)
            {
                this.IsDestroyed = true;
            }

            if(this.IsDestroyed)
            {
                TileDestructionEffect effect = new TileDestructionEffect(gameContentManager.TextureRepository, this.Position.PositionX + (TileSet.DRAW_SIZE/2), this.Position.PositionY + (TileSet.DRAW_SIZE/2));
                gameState.ActiveEffects.Add(effect);

                destructionStart = 0;

                if(this.SetVariableToTrueUponDestruction.Equals("") == false)
                {
                    gameState.VariableState.SetVariable(this.SetVariableToTrueUponDestruction, true);
                }
            }
        }

    }
}
