﻿using GameResources.Logic.EnemyMotion;
using GameResources.Logic.Physics;
using GameResources.Manager;
using GameResources.Models.Characters;
using GameResources.Models.Effects;
using GameResources.Models.Game;
using GameResources.Models.Physics;
using GameResources.Models.Weapons;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System;
using System.Collections.Generic;


namespace GameResources.Models.World
{
    public class Enemy : IBattleObject
    {
        public int Key { get; set; }
        public string Name { get; set; }
        public CharacterStats CharacterStats { get; set; }
        public WorldPosition Position { get; set; }
        public Motion Motion { get; set; }
        public IList<Texture2D> Textures { get; private set; }
        public bool IsScrewable { get; set; }
        public bool IsBombable { get; set; }
        public bool IsFreezeable { get; set; }
        public bool IsBoss { get; set; }
        public bool IsExplosive { get; set; }
        public bool IsImmediatelyFreezable { get; set; }
        public bool IsFrozen { get; set; }
        public string OnlyShowIfVariableIsFalse { get; set; }
        public string SetVariableToTrueUponDeath { get; set; }
        public bool CanShoot { get; set; }
        public int? EnemyKeyForShooting { get; set; }
        public double ShootSpeed { get; set; }
        public double ItemDropChance { get; set; }


        public bool IsExpired { get; set; }
        public int Attack { get; set; }
        public IEnemyMotion MotionType { get; set; }

        private int activeTexture;
        private long lastAnimationUpdate;
        private long lastMoveUpdate;
        private long lastShoot;

        private bool IsDamaged;
        private long lastDamageUpdate;
        private long freezeStart;


        public Enemy()
        {
            this.Textures = new List<Texture2D>();
            this.activeTexture = 0;
            this.Motion = new Motion();
            this.Position = new WorldPosition();
            this.CharacterStats = new CharacterStats();
            this.IsExpired = false;
            this.IsDamaged = false;
            this.IsFrozen = false;
            this.IsExplosive = false;

            this.lastAnimationUpdate = 0;
            this.lastMoveUpdate = 0;
            this.lastShoot = 0;
            this.lastDamageUpdate = 0;
            this.freezeStart = 0;
        }


        public void Update(GameState gameState, IGameContentManager gameContentManager, long ticks)
        {
            lastAnimationUpdate += ticks;
            lastMoveUpdate += ticks;
            if (IsFrozen) freezeStart += ticks;
            if (IsDamaged) lastDamageUpdate += ticks;
            lastShoot += ticks;

            //variable check
            if(OnlyShowIfVariableIsFalse.Equals("") == false)
            {
                bool state = gameState.VariableState.GetVariable(this.OnlyShowIfVariableIsFalse);

                if (state == true) 
                {
                    this.IsExpired = true;
                    
                    DropPickups(gameState, gameContentManager);

                    return;
                }
            }

            //reset damaged indicator
            if (IsDamaged && (lastDamageUpdate >= (TimeSpan.TicksPerSecond * 0.1)))
            {
                IsDamaged = false;
                lastDamageUpdate = 0;
            }

            //reset freeze
            if(IsFrozen && (freezeStart >= (TimeSpan.TicksPerSecond * 10)))
            {
                IsFrozen = false;
                freezeStart = 0;
            }

            //animate
            if (IsFrozen == false)
            {
                if (lastAnimationUpdate >= (TimeSpan.TicksPerSecond * 0.2))
                {
                    activeTexture++;
                    if (activeTexture >= this.Textures.Count) activeTexture = 0;
                    lastAnimationUpdate = 0;
                }
            }

            //move
            if (IsFrozen == false)
            {
                if (lastMoveUpdate >= (TimeSpan.TicksPerSecond * 0.0085))
                {
                    this.MotionType.Do(ticks, gameState);

                    lastMoveUpdate = 0;
                }
            }

            //shoot
            if (CanShoot && (lastShoot >= (TimeSpan.TicksPerSecond * ShootSpeed)))
            {
                Shoot(gameState, gameContentManager);

                lastShoot = 0;
            }

            //explosive check
            if (IsExplosive)
            {
                CollisionDetection collisionDetection = new CollisionDetection(this, gameState.GetActiveWorldMapCell(), gameState);
                if (collisionDetection.HasCollidedInWorldAtPosition((int)(this.Position.PositionX + this.Motion.VelocityX), (int)(this.Position.PositionY + this.Motion.VelocityY)))
                {
                    this.IsExpired = true;

                    SmallEnemyDeathEffect deathEffect = new SmallEnemyDeathEffect(gameContentManager.TextureRepository, gameContentManager.SoundEffectRepository, this.Position.PositionX, this.Position.PositionY - this.GetHeight() / 2);
                    gameState.ActiveEffects.Add(deathEffect);
                }
            }
        }


        public void Draw(SpriteBatch spriteBatch, long dx, long dy)
        {
            Texture2D texture = this.Textures[activeTexture];
            Rectangle rect = new Rectangle((int)(Position.PositionX - dx - (GetWidth() / 2)), (int)(Position.PositionY - dy - GetHeight()), GetWidth(), GetHeight());
            Color color = Color.White;

            if (IsDamaged) color = Color.Red;
            else if (IsFrozen) color = new Color(64, 64, 255);

            if(this.Motion.VelocityX < 0) spriteBatch.Draw(texture, rect, null, color, 0f, new Vector2(0, 0), SpriteEffects.FlipHorizontally, 0);
            else spriteBatch.Draw(texture, rect, color);
        }


        private void Shoot(GameState gameState, IGameContentManager gameContentManager)
        {
            if (this.EnemyKeyForShooting == null) return;

            int shootingEnemyKey = this.EnemyKeyForShooting.Value;
            long px = this.Position.PositionX;
            long py = this.Position.PositionY;

            Enemy shootingEnemy = gameContentManager.EnemyRepository.GetContentByKey(shootingEnemyKey);
            shootingEnemy.Position.PositionX = px;
            shootingEnemy.Position.PositionY = py;

            CollisionDetection collisionDetection = new CollisionDetection(shootingEnemy, gameState.GetActiveWorldMapCell(), gameState);
            if(collisionDetection.HasCollidedInWorld())
            {
                if (collisionDetection.HasCollidedInWorldAtPosition(px - shootingEnemy.GetWidth(), py)) shootingEnemy.Position.PositionX += shootingEnemy.GetWidth();
                if (collisionDetection.HasCollidedInWorldAtPosition(px + shootingEnemy.GetWidth(), py)) shootingEnemy.Position.PositionX += -shootingEnemy.GetWidth();

                if (collisionDetection.HasCollidedInWorldAtPosition(px, py - shootingEnemy.GetHeight())) shootingEnemy.Position.PositionY += shootingEnemy.GetHeight();
                if (collisionDetection.HasCollidedInWorldAtPosition(px, py + shootingEnemy.GetHeight())) shootingEnemy.Position.PositionY += -shootingEnemy.GetHeight();
            }


            gameState.ActiveEnemies.Add(shootingEnemy);
        }


        public Rectangle GetCollisionAreaAtPosition(long x, long y)
        {
            int w = GetWidth();
            int h = GetHeight();
            
            Rectangle area = new Rectangle((int)(x - (w / 2)), (int)(y - h), w, h);

            return area;
        }


        public Rectangle GetCurrentCollisionArea()
        {
            return GetCollisionAreaAtPosition(this.Position.PositionX, this.Position.PositionY);
        }


        public int GetWidth()
        {
            return this.Textures[activeTexture].Width * 2;
        }


        public int GetHeight()
        {
            return this.Textures[activeTexture].Height * 2;
        }


        public void AttemptDamage(DamageType damageType, WeaponModifier modifier, int damage, GameState gameState, IGameContentManager gameContentManager)
        {
            if (this.IsExpired) return;
            
            if((IsBombable == false) && (damageType == DamageType.Bomb || damageType == DamageType.SuperBomb))
            {
                return;
            }

            if((IsBoss) && (damageType == DamageType.Normal))
            {
                return;
            }

            bool doDamage = true;
            bool doFreeze = false;

            //freeze logic
            if(modifier.IsIce)
            {
                if ((IsFrozen == false) && IsFreezeable)
                {
                    if (IsImmediatelyFreezable)
                    {
                        doFreeze = true;
                    }
                    else if((IsImmediatelyFreezable == false) && ((this.CharacterStats.Health - damage) <= 0))
                    {
                        doFreeze = true;
                    }
                }
            }

            //damage logic
            if(doFreeze)
            {
                IsFrozen = true;
                freezeStart = 0;
                doDamage = false;
            }

            if (doDamage)
            {
                this.CharacterStats.Health -= damage;

                IsDamaged = true;
                lastDamageUpdate = 0;
            }

            //play injure sound
            var s = gameContentManager.SoundEffectRepository.GetContentByName(@"soundeffects\Enemy Hit").CreateInstance();
            s.Volume = 0.5f;
            s.Play();

            if (this.CharacterStats.Health < 0)
            {
                //create death effect
                SmallEnemyDeathEffect deathEffect = new SmallEnemyDeathEffect(gameContentManager.TextureRepository, gameContentManager.SoundEffectRepository, this.Position.PositionX, this.Position.PositionY - this.GetHeight() / 2);
                gameState.ActiveEffects.Add(deathEffect);

                if (this.SetVariableToTrueUponDeath.Equals("") == false)
                {
                    gameState.VariableState.SetVariable(this.SetVariableToTrueUponDeath, true);
                }

                this.IsExpired = true;

                DropPickups(gameState, gameContentManager);
            }
        }


        private void DropPickups(GameState gameState, IGameContentManager gameContentManager)
        {
            Random rand = new Random();
            double chanceToDrop = rand.NextDouble();

            Item item = null;

            bool[] needs = new bool[5];
            needs[0] = true;
            needs[1] = true;
            needs[2] = (gameState.Player.Inventory.MissilesMax != 0) && (gameState.Player.Inventory.Missiles < gameState.Player.Inventory.MissilesMax);
            needs[3] = (gameState.Player.Inventory.SuperMissilesMax != 0) && (gameState.Player.Inventory.SuperMissiles < gameState.Player.Inventory.SuperMissilesMax);
            needs[4] = (gameState.Player.Inventory.SuperBombsMax != 0) && (gameState.Player.Inventory.SuperBombs < gameState.Player.Inventory.SuperBombsMax);

            if (chanceToDrop >= (1-ItemDropChance))
            {
                bool need = false;
                int choice = 0;

                while (need == false)
                {
                    choice = rand.Next(0, 5);

                    need = needs[choice];
                }

                if(choice == 0) item = new Item(ItemType.PickupHealthLarge, gameContentManager, this.Position.PositionX - (this.GetWidth() / 2), this.Position.PositionY - this.GetHeight());
                else if (choice == 1) item = new Item(ItemType.PickupHealthSmall, gameContentManager, this.Position.PositionX - (this.GetWidth() / 2), this.Position.PositionY - this.GetHeight());
                else if (choice == 2) item = new Item(ItemType.PickupMissiles, gameContentManager, this.Position.PositionX - (this.GetWidth() / 2), this.Position.PositionY - this.GetHeight());
                else if (choice == 3) item = new Item(ItemType.PickupSuperMissiles, gameContentManager, this.Position.PositionX - (this.GetWidth() / 2), this.Position.PositionY - this.GetHeight());
                else if (choice == 4) item = new Item(ItemType.PickupSuperBombs, gameContentManager, this.Position.PositionX - (this.GetWidth() / 2), this.Position.PositionY - this.GetHeight());
            }

            if (item != null) gameState.ActiveItems.Add(item);
        }

    }
}
