﻿using System.Collections.Generic;
using System.Text;


namespace RawResources.Models
{
    public class WorldMapCellDefinition
    {
        public static int NUM_TILES_HIGH = 18;
        public static int NUM_TILES_WIDE = 32;

        public int WorldKey { get; set; }
        public int WorldMapX { get; set; }
        public int WorldMapY { get; set; }
        public int WorldMapWidth { get; set; }
        public int WorldMapHeight { get; set; }
        public int SongKey { get; set; }
        public int? ParallaxBackgroundTextureKey { get; set; }
        public bool IsHeated { get; set; }
        public int WaterLevel { get; set; }
        public IDictionary<int, IDictionary<int, int?>> TileSetKeysLayer1 { get; }
        public IDictionary<int, IDictionary<int, int?>> TileSetKeysLayer2 { get; }
        public IList<DestructableBlockPlacementDefinition> DestructableBlockPlacements { get; }
        public IList<DoorPlacementDefinition> DoorPlacements { get; }
        public IList<TeleporterDefinition> TeleporterPlacements { get; }
        public IList<ItemPlacementDefinition> ItemPlacements { get; set; }
        public IList<SaverDefinition> SaverPlacements { get; }
        public IList<EnemyPlacementDefinition> EnemyPlacements { get; }
        public IList<VictoryDefinition> VictoryPlacements { get; }


        public WorldMapCellDefinition()
        {
            TileSetKeysLayer1 = new Dictionary<int, IDictionary<int, int?>>();
            TileSetKeysLayer2 = new Dictionary<int, IDictionary<int, int?>>();
            DestructableBlockPlacements = new List<DestructableBlockPlacementDefinition>();
            DoorPlacements = new List<DoorPlacementDefinition>();
            TeleporterPlacements = new List<TeleporterDefinition>();
            ItemPlacements = new List<ItemPlacementDefinition>();
            SaverPlacements = new List<SaverDefinition>();
            EnemyPlacements = new List<EnemyPlacementDefinition>();
            VictoryPlacements = new List<VictoryDefinition>();
            IsHeated = false;
            WaterLevel = 0;
        }


        public WorldMapCellDefinition(int wk, int x, int y, int w, int h, int s, int? p) : this()
        {
            this.WorldKey = wk;
            this.WorldMapX = x;
            this.WorldMapY = y;
            this.WorldMapWidth = w;
            this.WorldMapHeight = h;
            this.SongKey = s;
            this.ParallaxBackgroundTextureKey = p;

            GenerateTiles();
        }


        private void GenerateTiles()
        {
            int numTilesHigh = NUM_TILES_HIGH; 
            int numTilesWide = NUM_TILES_WIDE; 

            for(int x = 0; x < (numTilesWide * this.WorldMapWidth); x++)
            {
                for(int y = 0; y < (numTilesHigh * this.WorldMapHeight); y++)
                {
                    AddTile(x, y, null, null);
                }
            }
        }


        public EnemyPlacementDefinition GetEnemyPlacementAtPosition(int x, int y)
        {
            foreach (EnemyPlacementDefinition placement in this.EnemyPlacements)
            {
                if (placement.X == x && placement.Y == y) return placement;
            }

            return null;
        }


        public SaverDefinition GetSaverPlacementAtPosition(int x, int y)
        {
            foreach (SaverDefinition placement in this.SaverPlacements)
            {
                if (placement.X == x && placement.Y == y) return placement;
            }

            return null;
        }


        public DestructableBlockPlacementDefinition GetDestructableBlockPlacementAtPosition(int x, int y)
        {
            foreach(DestructableBlockPlacementDefinition placement in this.DestructableBlockPlacements)
            {
                if (placement.X == x && placement.Y == y) return placement;
            }

            return null;
        }


        public ItemPlacementDefinition GetItemPlacementAtPosition(int x, int y)
        {
            foreach (ItemPlacementDefinition placement in this.ItemPlacements)
            {
                if (placement.X == x && placement.Y == y) return placement;
            }

            return null;
        }


        public DoorPlacementDefinition GetDoorPlacementAtPosition(int x, int y)
        {
            foreach (DoorPlacementDefinition placement in this.DoorPlacements)
            {
                if (placement.X == x && placement.Y == y) return placement;
            }

            return null;
        }


        public TeleporterDefinition GetTeleporterPlacementAtPosition(int x, int y)
        {
            foreach (TeleporterDefinition placement in this.TeleporterPlacements)
            {
                if (placement.X == x && placement.Y == y) return placement;
            }

            return null;
        }


        public VictoryDefinition GetVictoryPlacementAtPosition(int x, int y)
        {
            foreach (VictoryDefinition placement in this.VictoryPlacements)
            {
                if (placement.X == x && placement.Y == y) return placement;
            }

            return null;
        }


        public void AddTile(int x, int y, int? tileSetKey1, int? tileSetKey2)
        {
            if (TileSetKeysLayer1.ContainsKey(x) == false) TileSetKeysLayer1.Add(x, new Dictionary<int, int?>());
            if (TileSetKeysLayer2.ContainsKey(x) == false) TileSetKeysLayer2.Add(x, new Dictionary<int, int?>());

            TileSetKeysLayer1[x].Add(y, tileSetKey1);
            TileSetKeysLayer2[x].Add(y, tileSetKey2);
        }


        public int GetMaxTileX()
        {
            return TileSetKeysLayer1.Count;
        }


        public int GetMaxTileY()
        {
            int max = 0;

            if (TileSetKeysLayer1.Count > 0)
            {
                max = TileSetKeysLayer1[0].Count;
            }

            return max;
        }


        public string GetCSV()
        {
            StringBuilder builder = new StringBuilder();
            bool firstCell = false;

            builder.AppendLine("Cell|" + this.WorldMapX + "|" + this.WorldMapY + "|" + this.WorldMapWidth + "|" + this.WorldMapHeight + "|" + this.SongKey + "|" + this.ParallaxBackgroundTextureKey + "|" + this.IsHeated + "|" + this.WaterLevel);

            //tiles
            foreach (int yKey in this.TileSetKeysLayer1[0].Keys)
            {
                StringBuilder line = new StringBuilder();
                firstCell = true;

                foreach (int xKey in this.TileSetKeysLayer1.Keys)
                {
                    int? tile1 = this.TileSetKeysLayer1[xKey][yKey];
                    int? tile2 = this.TileSetKeysLayer2[xKey][yKey];

                    if (firstCell == false)
                    {
                        line.Append(",");
                    }

                    if (tile1 != null) line.Append(tile1);

                    if (tile2 != null)
                    {
                        line.Append("|");
                        line.Append(tile2);
                    }

                    firstCell = false;
                }

                builder.AppendLine(line.ToString());
            }

            //destructable blocks
            foreach(DestructableBlockPlacementDefinition block in this.DestructableBlockPlacements)
            {
                builder.AppendLine("Block|" + block.ToString());
            }

            //doors
            foreach(DoorPlacementDefinition door in this.DoorPlacements)
            {
                builder.AppendLine("Door|" + door.ToString());
            }

            //teleports
            foreach(TeleporterDefinition teleporter in this.TeleporterPlacements)
            {
                builder.AppendLine("Teleporter|" + teleporter.ToString());
            }

            //item placements
            foreach(ItemPlacementDefinition item in this.ItemPlacements)
            {
                builder.AppendLine("Item|" + item.ToString());
            }

            //enemy placements
            foreach(EnemyPlacementDefinition enemy in this.EnemyPlacements)
            {
                builder.AppendLine("Enemy|" + enemy.ToString());
            }

            //savers
            foreach (SaverDefinition saver in this.SaverPlacements)
            {
                builder.AppendLine("Saver|" + saver.ToString());
            }

            //victory
            foreach (VictoryDefinition saver in this.VictoryPlacements)
            {
                builder.AppendLine("Victory|" + saver.ToString());
            }

            return builder.ToString();
        }


    }
}
