﻿using GameResources.Models.Game;
using GameResources.Models.Samus;
using GameResources.Models.World;
using GameResources.Repository;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using RawResources.Models;
using System.Collections.Generic;


namespace UserInterface.Control
{
    public class MiniMap : IControl
    {
        public IList<IControl> Controls { get; set; }
        public bool IsSelected { get; set; }
        public bool IsSelectable { get; set; }
        public Rectangle Location { get; set; }
        public GameState GameState { get; set; }

        private Texture2D mapCell;
        private Texture2D mapPlayer;
        private Texture2D background;
        private SpriteFont font;
        private int numCells;
        private Color backgroundMask;


        public MiniMap(IGameNamedContentRepository<Texture2D> textureRepository, IGameNamedContentRepository<SpriteFont> fontRepository, int numCells, Color backgroundMask)
        {
            this.backgroundMask = backgroundMask;
            this.Controls = new List<IControl>();
            this.numCells = numCells;

            mapCell = textureRepository.GetContentByName(@"textures\ui\minimap-room");
            mapPlayer = textureRepository.GetContentByName(@"textures\ui\minimap-player");
            background = textureRepository.GetContentByName(@"textures\white");
            font = fontRepository.GetContentByName(@"fonts\Normal");
        }

        public void Update(long ticks)
        {

        }

        public void Draw(SpriteBatch spriteBatch)
        {
            if (this.Location == default) return;
            if (this.GameState == null) return;

            int halfNumCells = numCells / 2;
            int gridSize = (this.Location.Width) / numCells;
            int playerSize = gridSize / 2;

            //background
            Rectangle backgroundRect = new Rectangle(this.Location.X, this.Location.Y, this.Location.Width, this.Location.Height);
            spriteBatch.Draw(background, backgroundRect, backgroundMask);

            //grid
            Player player = GameState.Player;
            int cellDX = (int)(player.Position.PositionX / (WorldMapCellDefinition.NUM_TILES_WIDE * TileSet.DRAW_SIZE));
            int cellDY = (int)(player.Position.PositionY / (WorldMapCellDefinition.NUM_TILES_HIGH * TileSet.DRAW_SIZE));
            int cellX = player.Position.WorldCellX + cellDX;
            int cellY = player.Position.WorldCellY + cellDY;

            int i = 0;
            int j = 0;

            for (int x = cellX - halfNumCells; x <= cellX + halfNumCells; x++)
            {
                for (int y = cellY - halfNumCells; y <= cellY + halfNumCells; y++)
                {
                    WorldMapCell cell = GetPositionCoveredByCell(x, y);

                    if (cell != null)
                    {
                        if (this.GameState.MapHistory.HasVisitedCell(cell))
                        {
                            Rectangle rect = new Rectangle(this.Location.X + (i * gridSize), this.Location.Y + (j * gridSize), gridSize, gridSize);
                            spriteBatch.Draw(mapCell, rect, Color.White);

                            DrawSides(spriteBatch, cell, x, y, rect);

                            if(cell.HasSave())
                            {
                                spriteBatch.DrawString(font, "S", new Vector2(rect.X, rect.Y), Color.White);
                            }
                        }
                    }

                    if (x == cellX && y == cellY)
                    {
                        Rectangle rect = new Rectangle(this.Location.X + (i * gridSize) + (playerSize / 2), this.Location.Y + (j * gridSize) + (playerSize / 2), playerSize, playerSize);
                        spriteBatch.Draw(mapPlayer, rect, Color.White);
                    }

                    j++;
                }

                j = 0;
                i++;
            }
        }


        private void DrawSides(SpriteBatch spriteBatch, WorldMapCell cell, int wx, int wy, Rectangle rect)
        {
            //top
            if (cell.WorldMapY == wy)
            {
                for (int x = rect.X; x < (rect.X + rect.Width); x++) spriteBatch.Draw(background, new Rectangle(x, rect.Y, 1, 1), Color.White);
            }

            //bottom
            int cellBottom = cell.WorldMapHeight == 1 ? cell.WorldMapY : cell.WorldMapY + cell.WorldMapHeight -1;
            if(cellBottom == wy)
            {
                for (int x = rect.X; x < (rect.X + rect.Width); x++) spriteBatch.Draw(background, new Rectangle(x, rect.Y + rect.Height - 1, 1, 1), Color.White);
            }

            //left
            if (cell.WorldMapX == wx)
            {
                for (int y = rect.Y; y < (rect.Y + rect.Height); y++) spriteBatch.Draw(background, new Rectangle(rect.X, y, 1, 1), Color.White);
            }

            //right
            int cellRight = cell.WorldMapWidth == 1 ? cell.WorldMapX : cell.WorldMapX + cell.WorldMapWidth -1;
            if(cellRight == wx)
            {
                for (int y = rect.Y; y < (rect.Y + rect.Height); y++) spriteBatch.Draw(background, new Rectangle(rect.X + rect.Width - 1, y, 1, 1), Color.White);
            }
        }


        private WorldMapCell GetPositionCoveredByCell(int wx, int wy)
        {
            WorldMap map = this.GameState.GetActiveWorldMap();
            if (map == null) return null;

            WorldMapCell cell = map.GetCellAtLocation(wx, wy);

            return cell;
        }



        public void Press()
        {

        }
    }
}
