﻿using System;
using System.Numerics;
using GameEngine.Audio;
using GameEngine.Graphics;
using GameEngine.Graphics.Models;
using GameEngine.Input;

namespace FirstPersonShooter.Controls
{
	public class ButtonControl : IControl
	{
        public bool IsFocused { get; set; }
        public bool IsClicked { get; set; }
        public delegate void ClickEvent();
        public event ClickEvent? Click;

        private readonly SoundManager soundManager;
        private readonly GraphicsDevice graphicsDevice;
        private readonly QuadHelper quadHelper;
        private readonly float depth;

        private Texture? backgroundTexture;
        private Texture? activeBackgroundTexture;
        private Texture? clickedBackgroundTexture;
        private Texture? borderTexture;
        private Texture renderedText;
        private Vector2 upperLeftCorner;
        private Vector2 lowerRightCorner;
        private Quad? textQuad;
        private Quad? borderQuad;
        private Quad? backgroundQuad;

        private int clickTicks;


        public ButtonControl(MyGame game, string text, Vector2 upperLeftCorner, Vector2 lowerRightCorner, float depth)
		{
            this.IsFocused = false;
            this.IsClicked = false;
            this.soundManager = game.SoundManager;
            this.graphicsDevice = game.GraphicsDevice;
            this.quadHelper = new QuadHelper(game.GraphicsDevice);

            this.depth = depth;
            this.upperLeftCorner = upperLeftCorner;
            this.lowerRightCorner = lowerRightCorner;

            this.renderedText = game.FontRenderer.RenderString("kongtext", text, 16, 154, 200, 232);
            this.backgroundTexture = game.TextureManager.GetTextureByName("graphics/ui/button-background.png");
            this.activeBackgroundTexture = game.TextureManager.GetTextureByName("graphics/ui/button-background-active.png");
            this.clickedBackgroundTexture = game.TextureManager.GetTextureByName("graphics/ui/button-background-clicked.png");
            this.borderTexture = game.TextureManager.GetTextureByName("graphics/ui/button-border.png");

            this.clickTicks = 0;

            SetBorderQuad();
            SetBackgroundQuad();
            SetTextQuad();
        }


        private void SetBorderQuad()
        {
            if (borderTexture != null)
            {
                this.borderQuad = quadHelper.MakeQuadForUIFromAbsolute(borderTexture, upperLeftCorner, lowerRightCorner, depth + 0.1f);
            }
        }


        private void SetBackgroundQuad()
        {
            Texture? texture = null;

            if(this.IsClicked)
            {
                texture = clickedBackgroundTexture;
            }
            else if (this.IsFocused)
            {
                texture = activeBackgroundTexture;
            }
            else
            {
                texture = backgroundTexture;
            }

            if (texture != null)
            {
                float x1 = upperLeftCorner.X + 1;
                float y1 = upperLeftCorner.Y - 1;
                Vector2 v1 = new Vector2(x1, y1);

                float x2 = lowerRightCorner.X - 1;
                float y2 = lowerRightCorner.Y + 1;
                Vector2 v2 = new Vector2(x2, y2);
                this.backgroundQuad = quadHelper.MakeQuadForUIFromAbsolute(texture, v1, v2, depth);
            }
        }


        private void SetTextQuad()
        {
            float cx = upperLeftCorner.X + ((lowerRightCorner.X - upperLeftCorner.X) / 2);
            float cy = lowerRightCorner.Y + ((upperLeftCorner.Y - lowerRightCorner.Y) / 2);

            float x1 = cx - (renderedText.Width / 2);
            float y1 = cy - (renderedText.Height / 2);
            Vector2 v1 = new Vector2(x1, y1);

            float x2 = cx + (renderedText.Width / 2);
            float y2 = cy + (renderedText.Height / 2);
            Vector2 v2 = new Vector2(x2, y2);

            this.textQuad = quadHelper.MakeQuadForUIFromAbsolute(renderedText, v1, v2, depth - 0.1f);
        }


        public void Draw()
        {
            if (this.borderQuad != null) this.graphicsDevice.DrawQuadRelative(this.borderQuad);
            if (this.backgroundQuad != null) this.graphicsDevice.DrawQuadRelative(this.backgroundQuad);
            if (this.textQuad != null) this.graphicsDevice.DrawQuadRelative(this.textQuad);
        }


        public void Update(InputState inputState)
        {
            SetBackgroundQuad();

            if (IsFocused && !IsClicked)
            {
                if (inputState.Use || inputState.Jump || inputState.Shoot)
                {
                    this.IsClicked = true;
                    this.clickTicks = 0;
                }
            }

            if(this.IsClicked)
            {
                if (clickTicks == 0)
                {
                    //play sound
                    Sound? sound = this.soundManager.GetSoundByName("sounds/button-click.wav");
                    if (sound != null) sound.Play();
                }

                clickTicks++;

                if(clickTicks >= 12)
                {
                    this.IsClicked = false;

                    //invoke event
                    if (this.Click != null) Click.Invoke();
                }
            }
        }


        public bool ContainsPosition(Vector2 point)
        {
            if((point.X >= upperLeftCorner.X) && (point.X <= lowerRightCorner.X))
            {
                if((point.Y >= lowerRightCorner.Y) && (point.Y <= upperLeftCorner.Y))
                {
                    return true;
                }
            }

            return false;
        }

    }
}

