﻿using System.Numerics;
using FirstPersonShooter.Models;
using GameEngine.Graphics;
using RawResources.Models.Items;

namespace FirstPersonShooter.Logic
{
	public class BulletLauncher
	{
		private readonly GameState gameState;
        private readonly Texture? bulletTexture;
        private readonly Texture? fireBallTexture;
        private readonly Texture? lightningBallTexture;
        private readonly Texture? magicBallTexture;
        private readonly Texture? redLaserTexture1;
        private readonly Texture? redLaserTexture2;
        private readonly Texture? redLaserTexture3;


        public BulletLauncher(GameState gameState, TextureManager textureManager)
		{
			this.gameState = gameState;

            this.bulletTexture = textureManager.GetTextureByName("graphics/weapons/bullet.png");
            this.fireBallTexture = textureManager.GetTextureByName("graphics/weapons/fireball.png");

            this.lightningBallTexture = textureManager.GetTextureByName("graphics/weapons/lightingball.png");
            this.magicBallTexture = textureManager.GetTextureByName("graphics/weapons/magicball.png");

            this.redLaserTexture1 = textureManager.GetTextureByName("graphics/weapons/red-laser-1.png");
            this.redLaserTexture2 = textureManager.GetTextureByName("graphics/weapons/red-laser-2.png");
            this.redLaserTexture3 = textureManager.GetTextureByName("graphics/weapons/red-laser-3.png");
        }


		public void ShootWeapon(ICollisionObject shooter, ShootingTypes shootingType, int damage, Vector3 origin, Vector3 direction)
		{
            if (this.bulletTexture == null) return;
            if (this.fireBallTexture == null) return;
            if (this.redLaserTexture1 == null) return;
            if (this.redLaserTexture2 == null) return;
            if (this.redLaserTexture3 == null) return;
            if (this.lightningBallTexture == null) return;
            if (this.magicBallTexture == null) return;

            Level? activeLevel = this.gameState.ActiveLevel;
			if (activeLevel == null) return;

			if(shootingType == ShootingTypes.Gun_SingleShotAccurate)
			{
                float speed = 50f;
				float absoluteNoise = 0.0125f;

				Vector3 noise = GenerateNoise(absoluteNoise);
				Vector3 newDirection = direction + noise;

                Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.bulletTexture, new Texture[0], 1, 120, false);
                activeLevel.Bullets.Add(bullet);
            }
            else if (shootingType == ShootingTypes.BulletTurret)
            {
                float speed = 20f;
                float absoluteNoise = 0.02f;

                Vector3 noise = GenerateNoise(absoluteNoise);
                Vector3 newDirection = direction + noise;

                Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.bulletTexture, new Texture[0], 2, 120, false);
                activeLevel.Bullets.Add(bullet);
            }
			else if (shootingType == ShootingTypes.Gun_SingleShotLoose)
            {
                float speed = 90f;
                float absoluteNoise = 0.03f;

                Vector3 noise = GenerateNoise(absoluteNoise);
                Vector3 newDirection = direction + noise;

                Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.bulletTexture, new Texture[0], 1, 120, false);
                activeLevel.Bullets.Add(bullet);
            }
			else if(shootingType == ShootingTypes.Knife)
			{
                Bullet bullet = new Bullet(shooter, origin, direction, 8, damage, this.bulletTexture, new Texture[0], 1, 1, false);
                activeLevel.Bullets.Add(bullet);
            }
            else if (shootingType == ShootingTypes.Rocket)
            {
                float speed = 15f;

                Bullet bullet = new Bullet(shooter, origin, direction, speed, damage, this.fireBallTexture, new Texture[0], 8, 120, true);
                activeLevel.Bullets.Add(bullet);
            }
			else if(shootingType == ShootingTypes.Gun_ShotgunSpread)
			{
                float speed = 40f;
                float absoluteNoise = 0.035f;
                float angleDegrees = 5f;

                Vector3[] directions = GetShotGunDirections(direction, angleDegrees);

                for(int i = 0; i < directions.Length; i++)
                {
                    Vector3 noise = GenerateNoise(absoluteNoise);
                    Vector3 newDirection = directions[i] + noise;

                    Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.bulletTexture, new Texture[0], 1, 20, false);
                    activeLevel.Bullets.Add(bullet);
                }
            }
            else if (shootingType == ShootingTypes.FireBall_SingleShotSlowAccurate)
            {
                float speed = 15f;
                float absoluteNoise = 0.01f;

                Vector3 noise = GenerateNoise(absoluteNoise);
                Vector3 newDirection = direction + noise;

                Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.fireBallTexture, new Texture[0], 8, 120, false);
                activeLevel.Bullets.Add(bullet);
            }
            else if (shootingType == ShootingTypes.Laser_SingleShotAccurate)
            {
                float speed = 15f;
                float absoluteNoise = 0.0125f;

                Vector3 noise = GenerateNoise(absoluteNoise);
                Vector3 newDirection = direction + noise;

                Texture[] trailTextures = new Texture[6];
                trailTextures[0] = redLaserTexture1;
                trailTextures[1] = redLaserTexture1;
                trailTextures[2] = redLaserTexture1;
                trailTextures[3] = redLaserTexture1;
                trailTextures[4] = redLaserTexture1;
                trailTextures[5] = redLaserTexture1;

                Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.redLaserTexture1, trailTextures, 1, 120, false);
                activeLevel.Bullets.Add(bullet);
            }
            else if (shootingType == ShootingTypes.LightningBall_Accurate)
            {
                float speed = 15f;
                float absoluteNoise = 0.01f;

                Vector3 noise = GenerateNoise(absoluteNoise);
                Vector3 newDirection = direction + noise;

                Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.lightningBallTexture, new Texture[0], 12, 120, false);
                activeLevel.Bullets.Add(bullet);
            }
            else if (shootingType == ShootingTypes.MagicBall_Accurate)
            {
                float speed = 15f;
                float absoluteNoise = 0.01f;

                Vector3 noise = GenerateNoise(absoluteNoise);
                Vector3 newDirection = direction + noise;

                Bullet bullet = new Bullet(shooter, origin, newDirection, speed, damage, this.magicBallTexture, new Texture[0], 12, 120, false);
                activeLevel.Bullets.Add(bullet);
            }
        }


		private Vector3 GenerateNoise(float absoluteNoise)
		{
			Random random = new Random();
			float x = (random.NextSingle() * (absoluteNoise * 2)) - absoluteNoise;
            float y = (random.NextSingle() * (absoluteNoise * 2)) - absoluteNoise;
            float z = (random.NextSingle() * (absoluteNoise * 2)) - absoluteNoise;

			return new Vector3(x, y, z);
        }


		private Vector3[] GetShotGunDirections(Vector3 direction, float angleDegrees)
		{
			Vector3[] vectors = new Vector3[9];

            float angleRads = (MathF.PI / 180f) * angleDegrees;

			vectors[0] = direction;
            vectors[1] = RotateUpDown(direction, angleRads);
            vectors[2] = RotateUpDown(direction, -angleRads);
            vectors[3] = RotateLeftRight(direction, angleRads);
            vectors[4] = RotateLeftRight(direction, -angleRads);

            vectors[5] = RotateLeftRight(RotateUpDown(direction, angleRads), angleRads); //lower right
            vectors[6] = RotateLeftRight(RotateUpDown(direction, angleRads), -angleRads); //lower left
            vectors[7] = RotateLeftRight(RotateUpDown(direction, -angleRads), angleRads); //upper right
            vectors[8] = RotateLeftRight(RotateUpDown(direction, -angleRads), -angleRads); //upper left

            return vectors;
		}


		private Vector3 RotateUpDown(Vector3 original, float angle)
		{
            Vector3 axis = Vector3.Cross(original, Vector3.UnitZ);

            Quaternion rotationQuaternion = Quaternion.CreateFromAxisAngle(axis, angle);

            return Vector3.Transform(original, rotationQuaternion);
        }


        private Vector3 RotateLeftRight(Vector3 original, float angle)
        {
            Vector3 axis = Vector3.UnitZ;

            Quaternion rotationQuaternion = Quaternion.CreateFromAxisAngle(axis, angle);

            return Vector3.Transform(original, rotationQuaternion);
        }

    }
}

