﻿using System.Numerics;
using FirstPersonShooter.Models;
using GameEngine.Audio;
using RawResources.Models.Items;

namespace FirstPersonShooter.Logic.EnemyBehaviours
{
	public class RoamingLaserEnemyBehaviour : EnemyBehaviour
	{
        private enum MovementStates { TowardPlayer, StrafeLeft, StrafeRight, BackFromPlayer, Stop, Random }
        private MovementStates movementState;
        private int movementChangeTicks;
        private readonly int TICKS_BETWEEN_MOVEMENT_CHANGES = 40;
        private Vector3 moveDirection;

        private int shootTicks;
        private readonly int TICKS_BETWEEN_SHOTS = 90;
        private readonly int PLAYER_RANGE = 2048;
        private readonly int MIN_RANGE_FROM_PLAYER = 160;


        public RoamingLaserEnemyBehaviour() : base()
		{
            this.movementState = MovementStates.Stop;
            this.moveDirection = new Vector3();
		}


        protected override void Act(Enemy enemy, MyGame game, GameState gameState)
        {
            Player? player = gameState.Player;
            if (player == null) return;

            KeepAliveIfPlayerIsAround(enemy, player);
            Move(enemy, player, gameState);

            Shoot(enemy, player, game, gameState);
        }


        private void KeepAliveIfPlayerIsAround(Enemy enemy, Player player)
        {
            if (this.state == States.Active)
            {
                if (IsPlayerInArea(enemy, player, PLAYER_RANGE))
                {
                    this.playerVisibleTicks = 0;
                }
            }
        }


        private void Move(Enemy enemy, Player player, GameState gameState)
        {
            
            int speed = 3;

            if (this.movementChangeTicks < TICKS_BETWEEN_MOVEMENT_CHANGES) this.movementChangeTicks++;

            if(this.movementChangeTicks >= TICKS_BETWEEN_MOVEMENT_CHANGES)
            {
                ChangeDirection(enemy, player);

                this.movementChangeTicks = GetRandomTick(0, TICKS_BETWEEN_MOVEMENT_CHANGES / 3);
            }

            bool move = true;

            if(this.movementState == MovementStates.TowardPlayer)
            {
                if (IsPlayerInArea(enemy, player, MIN_RANGE_FROM_PLAYER))
                {
                    move = false;
                }
            }

            if (move)
            {
                Vector3 movementForce = this.moveDirection * speed;

                enemy.Force = movementForce;

                Vector3 newPosition = gameState.Physics.GetNewPosition(enemy, true);

                enemy.Position = newPosition;
            }
        }


        private void ChangeDirection(Enemy enemy, Player player)
        {
            Random random = new Random();
            int rd = random.Next(0, 5);
            MovementStates newState = this.movementState;

            if (rd == 0) newState = MovementStates.TowardPlayer;
            else if (rd == 1) newState = MovementStates.StrafeLeft;
            else if (rd == 2) newState = MovementStates.StrafeRight;
            else if (rd == 3) newState = MovementStates.BackFromPlayer;
            else if (rd == 4) newState = MovementStates.Random;
            else newState = MovementStates.Stop;

            Vector3 newDirection = new Vector3(0);

            if (newState == MovementStates.TowardPlayer)
            {
                newDirection = GetDirectionToPlayer(enemy, player);
            }
            else if (newState == MovementStates.BackFromPlayer)
            {
                newDirection = GetDirectionToPlayer(enemy, player) * -1;
            }
            else if(newState == MovementStates.StrafeLeft)
            {
                float radians = 90f * MathF.PI / 180f;

                Vector3 directionToPlayer = GetDirectionToPlayer(enemy, player);
                Quaternion rotationQuaternion = Quaternion.CreateFromYawPitchRoll(0, 0, radians);

                newDirection = Vector3.Transform(directionToPlayer, rotationQuaternion);
            }
            else if (newState == MovementStates.StrafeRight)
            {
                float radians = -90f * MathF.PI / 180f;

                Vector3 directionToPlayer = GetDirectionToPlayer(enemy, player);
                Quaternion rotationQuaternion = Quaternion.CreateFromYawPitchRoll(0, 0, radians);

                newDirection = Vector3.Transform(directionToPlayer, rotationQuaternion);
            }
            else if (newState == MovementStates.Random)
            {
                float x = random.NextSingle();
                float y = random.NextSingle();

                newDirection = Vector3.Normalize(new Vector3(x, y, 0));
            }

            this.movementState = newState;
            this.moveDirection = newDirection;
        }


        private void Shoot(Enemy enemy, Player player, MyGame game, GameState gameState)
        {
            //shoot timer countdown
            if (this.shootTicks < TICKS_BETWEEN_SHOTS) this.shootTicks++;

            //shoot if able
            if (this.shootTicks >= TICKS_BETWEEN_SHOTS)
            {
                if (CanSeePlayer(enemy, gameState))
                {
                    Vector3 direction = GetDirectionToPlayer(enemy, player);
                    Vector3 origin = enemy.GetShootingOrigin(direction);

                    gameState.BulletLauncher.ShootWeapon(enemy, ShootingTypes.Laser_SingleShotAccurate, enemy.Damage, origin, direction);

                    Sound? sound = game.SoundManager.GetSoundByName("sounds/laser.wav");
                    if (sound != null) sound.Play();

                    //reset shoot timer
                    this.shootTicks = GetRandomTick(0, TICKS_BETWEEN_SHOTS / 3);
                }
            }
        }

    }
}

