﻿using GameEngine.Audio;
using GameEngine.Graphics;
using GameEngine.Graphics.Models;
using System.Numerics;

namespace FirstPersonShooter.Models
{
    public class Level
    {
        public int Key { get; set; }
        public IList<Geometry> Geometry { get; private set; }
        public IList<Light> Lights { get; private set; }
        public IList<Pickup> Pickups { get; private set; }
        public IList<Enemy> Enemies { get; private set; }
        public IList<Bullet> Bullets { get; private set; }
        public IList<Decal> Decals { get; private set; }
        public IList<Effect> Effects { get; private set; }
        public IList<TextPoint> TextPoints { get; private set; }
        public Light AmbientLight { get; set; }
        public Vector3 StartPosition { get; set; }
        public Sound? BackgroundMusic { get; set; }
        public IList<Door> Doors { get; private set; }
        public IList<Teleporter> Teleporters { get; private set; }
        public Texture? SkyBoxTexture { get; set; }

        public IDictionary<int, IDictionary<int, IList<Quad>>> Quad1stPassCache { get; private set; }
        public IDictionary<int, IDictionary<int, IList<Quad>>> Quad2ndPassCache { get; private set; }
        public IDictionary<int, IDictionary<int, IList<Geometry>>> GeometryCache { get; private set; }
        public IList<int> CacheXKeys { get; set; }
        public IList<int> CacheYKeys { get; set; }
        
        
        public Level()
        {
            this.Geometry = new List<Geometry>();
            this.Lights = new List<Light>();
            this.Pickups = new List<Pickup>();
            this.AmbientLight = new Light();
            this.Enemies = new List<Enemy>();
            this.Bullets = new List<Bullet>();
            this.Decals = new List<Decal>();
            this.StartPosition = new Vector3(0);
            this.Doors = new List<Door>();
            this.Effects = new List<Effect>();
            this.Teleporters = new List<Teleporter>();
            this.TextPoints = new List<TextPoint>();

            this.BackgroundMusic = null;
            this.Quad1stPassCache = new Dictionary<int, IDictionary<int, IList<Quad>>>();
            this.Quad2ndPassCache = new Dictionary<int, IDictionary<int, IList<Quad>>>();
            this.GeometryCache = new Dictionary<int, IDictionary<int, IList<Geometry>>>();
            this.CacheXKeys = new List<int>();
            this.CacheYKeys = new List<int>();
            this.Key = -1;
        }


        public void Update(MyGame game, GameState gameState)
        {
            foreach(Geometry geometry in this.Geometry)
            {
                geometry.Update();
            }

            foreach (Door door in this.Doors)
            {
                door.Update();
            }

            foreach (Pickup pickup in this.Pickups)
            {
                pickup.Update();
            }

            for (int i = 0; i < this.Effects.Count; i++)
            {
                Effect effect = this.Effects[i];

                effect.Update(game, gameState);

                if (effect.IsActive == false)
                {
                    this.Effects.RemoveAt(i);
                    i--;
                }
            }

            //System.Console.WriteLine("START----");
            //Player player = gameState.Player;
            for (int i = 0; i < this.Enemies.Count; i++)
            {
                Enemy enemy = this.Enemies[i];

                //double distance = Vector3.Distance(player.Position, enemy.Position);
                //DateTime start = DateTime.Now;

                enemy.Update(game, gameState);

                if (enemy.IsActive == false)
                {
                    this.Enemies.RemoveAt(i);
                    i--;
                }

                //DateTime end = DateTime.Now;

                //System.Console.WriteLine(enemy.Key.ToString("00") + " - " + distance.ToString("0,000.00") + " - " + (end - start).TotalMilliseconds.ToString("0.000"));
            }
            //System.Console.WriteLine("END----");

            for (int i = 0; i < this.Bullets.Count; i++)
            {
                Bullet bullet = this.Bullets[i];

                bullet.Update(game, gameState);

                if(bullet.IsActive == false)
                {
                    this.Bullets.RemoveAt(i);
                    i--;
                }
            }

            for(int i = 0; i < this.Decals.Count; i++)
            {
                Decal decal = this.Decals[i];

                decal.Update();

                if(decal.IsActive == false)
                {
                    this.Decals.RemoveAt(i);
                    i--;
                }
            }
        }


        public void SetCaches()
        {

            foreach(Geometry geometry in this.Geometry)
            {
                foreach(Quad quad in geometry.Quads)
                {
                    bool hasAlpha = quad.Texture.HasAlpha;
                    int x = (int)quad.Centroid.X;
                    int y = (int)quad.Centroid.Y;

                    //Keys
                    if (this.CacheXKeys.Contains(x) == false) this.CacheXKeys.Add(x);
                    if (this.CacheYKeys.Contains(y) == false) this.CacheYKeys.Add(y);

                    //Geometry Cache
                    if (this.GeometryCache.ContainsKey(x) == false) this.GeometryCache.Add(x, new Dictionary<int, IList<Geometry>>());
                    if (this.GeometryCache[x].ContainsKey(y) == false) this.GeometryCache[x].Add(y, new List<Geometry>());

                    if (this.GeometryCache[x][y].Contains(geometry) == false) this.GeometryCache[x][y].Add(geometry);


                    if (hasAlpha == false)
                    {
                        //Non Alpha Cache
                        if (this.Quad1stPassCache.ContainsKey(x) == false) this.Quad1stPassCache.Add(x, new Dictionary<int, IList<Quad>>());
                        if (this.Quad1stPassCache[x].ContainsKey(y) == false) this.Quad1stPassCache[x].Add(y, new List<Quad>());

                        this.Quad1stPassCache[x][y].Add(quad);
                    }
                    else
                    {
                        //Alpha Cache
                        if (this.Quad2ndPassCache.ContainsKey(x) == false) this.Quad2ndPassCache.Add(x, new Dictionary<int, IList<Quad>>());
                        if (this.Quad2ndPassCache[x].ContainsKey(y) == false) this.Quad2ndPassCache[x].Add(y, new List<Quad>());

                        this.Quad2ndPassCache[x][y].Add(quad);
                    }
                }
            }
        }


        public IList<Quad>? GetQuadsFromCache(int x, int y, int pass)
        {
            if (pass == 1)
            {
                if (this.Quad1stPassCache.ContainsKey(x) == false) return null;
                if (this.Quad1stPassCache[x].ContainsKey(y) == false) return null;

                return this.Quad1stPassCache[x][y];
            }
            else if (pass == 2)
            {
                if (this.Quad2ndPassCache.ContainsKey(x) == false) return null;
                if (this.Quad2ndPassCache[x].ContainsKey(y) == false) return null;

                return this.Quad2ndPassCache[x][y];
            }
            else
            {
                return null;
            }
        }


        public IList<Geometry>? GetGeometriesFromCache(int x, int y)
        {
            if (this.GeometryCache.ContainsKey(x) == false) return null;
            if (this.GeometryCache[x].ContainsKey(y) == false) return null;

            return this.GeometryCache[x][y];
        }

    }
}
