﻿using System;
using System.Numerics;
using FirstPersonShooter.Controls;
using FirstPersonShooter.Logic;
using FirstPersonShooter.Models;
using GameEngine.Input;

namespace FirstPersonShooter.Screen
{
	public class PauseScreen : IScreen
    {
        private readonly MyGame game;
        private GameStateManager gameStateManager;
        private PlayerCamera camera;
        private IList<IControl> controls;
        private IList<ButtonControl> buttons;


        public PauseScreen(MyGame game)
		{
            this.game = game;
            this.gameStateManager = new GameStateManager(this.game);
            this.camera = new PlayerCamera(game);

            this.controls = new List<IControl>();
            this.buttons = new List<ButtonControl>();

            Init();
        }


        private void Init()
        {
            int screenWidth = this.game.GraphicsDevice.ScreenWidth;
            int screenHeight = this.game.GraphicsDevice.ScreenHeight;
            int halfScreenWidth = screenWidth / 2;

            int buttonWidth = 180;
            int buttonHeight = 48;
            int buttonVertSpacing = 20;

            int x1 = 0;
            int y1 = 0;
            int x2 = 0;
            int y2 = 0;
            float depth = 0.9f;

            //background
            x1 = 0;
            y1 = screenHeight;
            x2 = screenWidth;
            y2 = 0;
            ImageControl backgroundImage = new ImageControl(this.game, "graphics/ui/hud-background.png", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            this.controls.Add(backgroundImage);

            int buttonVertStart = screenHeight - 96 - 96 - 64;
            depth = 0.5f;

            //resume game button
            x1 = halfScreenWidth - (buttonWidth / 2);
            y1 = buttonVertStart;
            x2 = halfScreenWidth + (buttonWidth / 2);
            y2 = y1 - buttonHeight;
            ButtonControl newGameButton = new ButtonControl(this.game, "Resume", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            newGameButton.Click += ResumeGame;
            this.controls.Add(newGameButton);
            this.buttons.Add(newGameButton);

            //save game button
            x1 = halfScreenWidth - (buttonWidth / 2);
            y1 = buttonVertStart - buttonHeight - buttonVertSpacing;
            x2 = halfScreenWidth + (buttonWidth / 2);
            y2 = y1 - buttonHeight;
            ButtonControl loadGameButton = new ButtonControl(this.game, "Save Game", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            loadGameButton.Click += SaveGame;
            this.controls.Add(loadGameButton);
            this.buttons.Add(loadGameButton);

            //leave game button
            x1 = halfScreenWidth - (buttonWidth / 2);
            y1 = buttonVertStart - buttonHeight * 2 - buttonVertSpacing * 4;
            x2 = halfScreenWidth + (buttonWidth / 2);
            y2 = y1 - buttonHeight;
            ButtonControl exitGameButton = new ButtonControl(this.game, "Main Menu", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            exitGameButton.Click += MainMenu;
            this.controls.Add(exitGameButton);
            this.buttons.Add(exitGameButton);

            depth = 0.00001f;

            //cursor
            MouseCursorControl mouseCursorControl = new MouseCursorControl(this.game, "graphics/ui/cursor.png", new Vector2(0, 0), depth);
            this.controls.Add(mouseCursorControl);
        }


        public void Activate()
        {
            this.game.InputDevice.SetMouseMode(MouseMode.UI);

            this.camera.GameState = this.game.ActiveGameState; 
        }


        public void Deactivate()
        {
            
        }


        public void Draw()
        {
            //game in the background to draw over
            //this.camera.Draw();

            for(int i = 0; i < this.controls.Count; i++)
            {
                this.controls[i].Draw();
            }
        }


        public void Update(InputState inputState)
        {
            //cursor input
            SelectBasedOnCursorIfMoved(inputState);

            //keyboard input
            if (this.game.CanDoCommonInput())
            {
                if (inputState.Up)
                {
                    MoveCursor(-1);
                    this.game.ResetCommonInput();
                }
                else if (inputState.Down)
                {
                    MoveCursor(1);
                    this.game.ResetCommonInput();
                }
                else if (inputState.Menu)
                {
                    ResumeGame();
                    return;
                }
            }

            //update all UI components
            for (int i = 0; i < this.controls.Count; i++)
            {
                this.controls[i].Update(inputState);
            }
        }


        private void SelectBasedOnCursorIfMoved(InputState inputState)
        {
            bool moved = false;

            if (inputState.MouseMovement.X != 0) moved = true;
            else if (inputState.MouseMovement.Y != 0) moved = true;

            if (moved)
            {
                for (int i = 0; i < this.buttons.Count; i++)
                {
                    if (this.buttons[i].ContainsPosition(inputState.MousePosition))
                    {
                        ButtonControl? activeButton = this.buttons.Where(x => x.IsFocused).FirstOrDefault();
                        if (activeButton != null) activeButton.IsFocused = false;

                        this.buttons[i].IsFocused = true;
                    }
                }
            }
        }


        private void MoveCursor(int c)
        {
            ButtonControl? activeButton = this.buttons.Where(x => x.IsFocused).FirstOrDefault();

            int i = -1;
            if (activeButton != null) i = this.buttons.IndexOf(activeButton);

            i += c;

            if (i < 0) i = this.buttons.Count - 1;
            else if (i >= this.buttons.Count) i = 0;

            if (activeButton != null) activeButton.IsFocused = false;

            this.buttons[i].IsFocused = true;
        }


        private void ResumeGame()
        {
            this.game.ResetCommonInput();
            this.game.ActivateScreen(ScreenTypes.Game);
        }


        private void SaveGame()
        {
            this.gameStateManager.Save();

            this.game.ResetCommonInput();
            this.game.ActivateScreen(ScreenTypes.Game);
        }


        private void MainMenu()
        {
            this.game.ResetCommonInput();
            this.game.ActivateScreen(ScreenTypes.MainMenu);
        }


    }
}

