﻿using System.Numerics;
using RawResources;
using RawResources.Models.Levels;
using RawResources.Models.Resources;


namespace GameEditor.Models
{
	public class Compiler
	{
        private readonly Session session;
        private Dictionary<int, Dictionary<int, List<QuadDefinition>>> quadsInLevel;
        private Dictionary<int, Dictionary<int, List<QuadDefinition>>> quadsInLevelThatBlockLight;


        public Compiler(Session session)
		{
            this.session = session;

            this.quadsInLevel = new Dictionary<int, Dictionary<int, List<QuadDefinition>>>();
            this.quadsInLevelThatBlockLight = new Dictionary<int, Dictionary<int, List<QuadDefinition>>>();
        }


		public void Compile()
		{
            if (this.session.RawContentManager == null) return;

            IRawContentManager rawContentManager = this.session.RawContentManager;

			IEnumerable<LevelDefinition> levels = rawContentManager.LevelRepository.GetAllContent();
            foreach (LevelDefinition level in levels)
			{
                if (this.session.LevelsNeedCompiling.ContainsKey(level.Key))
                {
                    DateTime start = DateTime.Now;

                    DeleteBadGeometry(level);

                    GenerateQuads(rawContentManager, level);

                    DateTime end = DateTime.Now;

                    double time = (end - start).TotalSeconds;

                    System.Console.Out.WriteLine(level.Name + "-TOTAL" + time.ToString("#,##0.00"));

                    quadsInLevel.Clear();
                    quadsInLevelThatBlockLight.Clear();

                    this.session.SetLevelNeedsCompilationStatus(level.Key, false);
                }
			}
		}


        private void DeleteBadGeometry(LevelDefinition level)
        {
            for(int i = 0; i < level.Geometry.Count; i++)
            {
                GeometryDefinition geometry = level.Geometry[i];

                (Vector3, Vector3) points = geometry.GetPoints();
                bool remove = false;

                if (points.Item1 == points.Item2) remove = true;
                else if (points.Item1.X - points.Item2.X == 0) remove = true;
                else if (points.Item1.Y - points.Item2.Y == 0) remove = true;
                else if (points.Item1.Z - points.Item2.Z == 0) remove = true;

                if (remove)
                {
                    level.Geometry.RemoveAt(i);
                    i--;
                }
            }
        }


		private void GenerateQuads(IRawContentManager rawContentManager, LevelDefinition level)
		{
            DateTime start = DateTime.Now;

            quadsInLevel.Clear();

            foreach (GeometryDefinition geometry in level.Geometry.Where(x => x.BlocksCollision == true && x.IsTeleporter == false))
			{
                GenerateFaces(rawContentManager, geometry);
            }

            foreach (GeometryDefinition geometry in level.Geometry.Where(x => x.BlocksCollision == false && x.IsTeleporter == false))
            {
                GenerateFaces(rawContentManager, geometry);
            }

            DateTime faceEnd = DateTime.Now;
            System.Console.Out.WriteLine(level.Name + " Generate Faces " + (faceEnd - start).TotalSeconds.ToString("#,##0.00"));

            DateTime removeOverlapEnd = DateTime.Now;
            System.Console.Out.WriteLine(level.Name + " Remove Overlap " + (removeOverlapEnd - faceEnd).TotalSeconds.ToString("#,##0.00"));

            AssignLighting(level);

            DateTime assignLightingEnd = DateTime.Now;
            System.Console.Out.WriteLine(level.Name + " Assign Lighting " + (assignLightingEnd - removeOverlapEnd).TotalSeconds.ToString("#,##0.00"));
        }


        private bool IsQuadFullyCoveredBy(IList<Vector3> points1, IList<Vector3> points2)
        {
            // Calculate the minimum and maximum coordinates of each square
            Vector3 square1Min = GetMinimumCoordinates(points1);
            Vector3 square1Max = GetMaximumCoordinates(points1);

            Vector3 square2Min = GetMinimumCoordinates(points2);
            Vector3 square2Max = GetMaximumCoordinates(points2);

            // Check if the first square is fully covered by the second square
            return square1Min.X >= square2Min.X && square1Min.Y >= square2Min.Y && square1Min.Z >= square2Min.Z &&
                   square1Max.X <= square2Max.X && square1Max.Y <= square2Max.Y && square1Max.Z <= square2Max.Z;
        }


        public Vector3 GetMinimumCoordinates(IList<Vector3> vertices)
        {
            Vector3 min = vertices[0];
            for (int i = 1; i < vertices.Count; i++)
            {
                min.X = Math.Min(min.X, vertices[i].X);
                min.Y = Math.Min(min.Y, vertices[i].Y);
                min.Z = Math.Min(min.Z, vertices[i].Z);
            }
            return min;
        }


        public Vector3 GetMaximumCoordinates(IList<Vector3> vertices)
        {
            Vector3 max = vertices[0];
            for (int i = 1; i < vertices.Count; i++)
            {
                max.X = Math.Max(max.X, vertices[i].X);
                max.Y = Math.Max(max.Y, vertices[i].Y);
                max.Z = Math.Max(max.Z, vertices[i].Z);
            }
            return max;
        }


        private bool AreSameQuadPoints(IList<Vector3> points1, IList<Vector3> points2)
        {
            if (points1[0] != points2[0]) return false;
            if (points1[1] != points2[1]) return false;
            if (points1[2] != points2[2]) return false;
            if (points1[3] != points2[3]) return false;

            return true;
        }


		private void GenerateFaces(IRawContentManager rawContentManager, GeometryDefinition geometry)
		{
            geometry.Quads.Clear();
            
            GraphicDefinition? texture = rawContentManager.GraphicRepository.GetContent(geometry.TextureKey);
			if (texture == null) return;

            (Vector3, Vector3) points = geometry.GetPoints();
			Vector3 p1 = points.Item1;
			Vector3 p2 = points.Item2;

            //NOTE: Quad Faces are generated based on 32 pixel textures - starting and ending on partials if needed (grid based)
            int gridSize = 32;
            int gx1 = (int)MathF.Floor(p1.X / gridSize) * gridSize;
            int gx2 = (int)MathF.Ceiling(p2.X / gridSize) * gridSize;
            int gy1 = (int)MathF.Floor(p1.Y / gridSize) * gridSize;
            int gy2 = (int)MathF.Ceiling(p2.Y / gridSize) * gridSize;
            int gz1 = (int)MathF.Floor(p1.Z / gridSize) * gridSize;
            int gz2 = (int)MathF.Ceiling(p2.Z / gridSize) * gridSize;

            int x1 = (int)p1.X;
            int x2 = (int)p2.X;
            int y1 = (int)p1.Y;
            int y2 = (int)p2.Y;
            int z1 = (int)p1.Z;
            int z2 = (int)p2.Z;

            int x, y, z;
            int dx, dy, dz;


            //TOP (X, Y @ Z1)
            x = x1;
            for (int gx = gx1; gx < gx2; gx += gridSize)
			{
                dx = gridSize;
                if (x > gx) dx = dx - (x - gx);
                if ((x+dx) > x2) dx = dx - (gx + gridSize - x2);

                y = y1;
                for (int gy = gy1; gy < gy2; gy += gridSize)
				{
                    dy = gridSize;
                    if (y > gy) dy = dy - (y - gy);
                    if ((y + dy) > y2) dy = dy - (gy + gridSize - y2);

                    QuadDefinition quad = new QuadDefinition();
					quad.Points[0] = new Vector3(x, y, p1.Z);
                    quad.Points[1] = new Vector3(x, y + dy, p1.Z);
                    quad.Points[2] = new Vector3(x + dx, y, p1.Z);
                    quad.Points[3] = new Vector3(x + dx, y + dy, p1.Z);

                    AddQuad(geometry, quad);

                    y += dy;
                }

                x += dx;
			}
            
            //BOTTOM (X, Y @ Z2)
            x = x1;
            for (int gx = gx1; gx < gx2; gx += gridSize)
            {
                dx = gridSize;
                if (x > gx) dx = dx - (x - gx);
                if ((x + dx) > x2) dx = dx - (gx + gridSize - x2);

                y = y1;
                for (int gy = gy1; gy < gy2; gy += gridSize)
                {
                    dy = gridSize;
                    if (y > gy) dy = dy - (y - gy);
                    if ((y + dy) > y2) dy = dy - (gy + gridSize - y2);

                    QuadDefinition quad = new QuadDefinition();
                    quad.Points[0] = new Vector3(x, y, p2.Z);
                    quad.Points[1] = new Vector3(x, y + dy, p2.Z);
                    quad.Points[2] = new Vector3(x + dx, y, p2.Z);
                    quad.Points[3] = new Vector3(x + dx, y + dy, p2.Z);

                    AddQuad(geometry, quad);

                    y += dy;
                }

                x += dx;
            }
            
            //NORTH (X, Z @ Y1)
            x = x1;
            for (int gx = gx1; gx < gx2; gx += gridSize)
            {
                dx = gridSize;
                if (x > gx) dx = dx - (x - gx);
                if ((x + dx) > x2) dx = dx - (gx + gridSize - x2);

                z = z1;
                for (int gz = gz1; gz < gz2; gz += gridSize)
				{
                    dz = gridSize;
                    if (z > gz) dz = dz - (z - gz);
                    if ((z + dz) > z2) dz = dz - (gz + gridSize - z2);
                    
                    QuadDefinition quad = new QuadDefinition();
					quad.Points[0] = new Vector3(x, p1.Y, z);
                    quad.Points[1] = new Vector3(x + dx, p1.Y, z);
                    quad.Points[2] = new Vector3(x, p1.Y, z + dz);
                    quad.Points[3] = new Vector3(x + dx, p1.Y, z + dz);

                    AddQuad(geometry, quad);

                    z += dz;
                }

                x += dx;
            }
            
            //SOUTH (X, Z @ Y2)
            x = x1;
            for (int gx = gx1; gx < gx2; gx += gridSize)
            {
                dx = gridSize;
                if (x > gx) dx = dx - (x - gx);
                if ((x + dx) > x2) dx = dx - (gx + gridSize - x2);

                z = z1;
                for (int gz = gz1; gz < gz2; gz += gridSize)
                {
                    dz = gridSize;
                    if (z > gz) dz = dz - (z - gz);
                    if ((z + dz) > z2) dz = dz - (gz + gridSize - z2);
                    
                    QuadDefinition quad = new QuadDefinition();
                    quad.Points[0] = new Vector3(x, p2.Y, z);
                    quad.Points[1] = new Vector3(x + dx, p2.Y, z);
                    quad.Points[2] = new Vector3(x, p2.Y, z + dz);
                    quad.Points[3] = new Vector3(x + dx, p2.Y, z + dz);

                    AddQuad(geometry, quad);

                    z += dz;
                }

                x += dx;
            }
            
            //WEST (Y, Z @ X1)
            y = y1;
            for (int gy = gy1; gy < gy2; gy += gridSize)
            {
                dy = gridSize;
                if (y > gy) dy = gridSize - (y - gy);
                if ((y + dy) > y2) dy = dy - (gy + gridSize - y2);

                z = z1;
                for (int gz = gz1; gz < gz2; gz += gridSize)
                {
                    dz = gridSize;
                    if (z > gz) dz = dz - (z - gz);
                    if ((z + dz) > z2) dz = dz - (gz + gridSize - z2);
                    
                    QuadDefinition quad = new QuadDefinition();
                    quad.Points[0] = new Vector3(p1.X, y, z);
                    quad.Points[1] = new Vector3(p1.X, y + dy, z);
                    quad.Points[2] = new Vector3(p1.X, y, z + dz);
                    quad.Points[3] = new Vector3(p1.X, y + dy, z + dz);

                    AddQuad(geometry, quad);

                    z += dz;
                }

                y += dy;
            }

            //EAST (Y, Z @ X2)
            y = y1;
            for (int gy = gy1; gy < gy2; gy += gridSize)
            {
                dy = gridSize;
                if (y > gy) dy = dy - (y - gy);
                if ((y + dy) > y2) dy = dy - (gy + gridSize - y2);

                z = z1;
                for (int gz = gz1; gz < gz2; gz += gridSize)
                {
                    dz = gridSize;
                    if (z > gz) dz = gridSize - (z - gz);
                    if ((z + dz) > z2) dz = dz - (gz + gridSize - z2);
                    
                    QuadDefinition quad = new QuadDefinition();
                    quad.Points[0] = new Vector3(p2.X, y, z);
                    quad.Points[1] = new Vector3(p2.X, y + dy, z);
                    quad.Points[2] = new Vector3(p2.X, y, z + dz);
                    quad.Points[3] = new Vector3(p2.X, y + dy, z + dz);

                    AddQuad(geometry, quad);

                    z += dz;
                }

                y += dy;
            }
            
        }


        private void AddQuad(GeometryDefinition geometry, QuadDefinition quad)
        {
            int x = (int)quad.Centroid.X;
            int y = (int)quad.Centroid.Y;
            int d = 16;

            if (geometry.IsDoor)
            {
                geometry.Quads.Add(quad);
                AddToQuadsThatBlockLight(x, y, quad);
                return;
            }

            bool canAdd = true;
            ICollection<QuadDefinition> quadsToSearch = GetQuadsToSeach(x, y, d);

            foreach (QuadDefinition quad2 in quadsToSearch)
            {
                if (AreSameQuadPoints(quad.Points, quad2.Points) || IsQuadFullyCoveredBy(quad.Points, quad2.Points))
                {
                    canAdd = false;
                    break;
                }
            }

            if (canAdd)
            {
                AddToQuadsInLevel(x, y, quad);

                if (geometry.BlocksLight)
                {
                    AddToQuadsThatBlockLight(x, y, quad);
                }

                geometry.Quads.Add(quad);
            }
        }


        private void AddToQuadsInLevel(int x, int y, QuadDefinition quad)
        {
            if (quadsInLevel.ContainsKey(x) == false)
            {
                quadsInLevel.Add(x, new Dictionary<int, List<QuadDefinition>>());
            }
            if (quadsInLevel[x].ContainsKey(y) == false)
            {
                quadsInLevel[x].Add(y, new List<QuadDefinition>());
            }
            quadsInLevel[x][y].Add(quad);
        }


        private void AddToQuadsThatBlockLight(int x, int y, QuadDefinition quad)
        {
            if (quadsInLevelThatBlockLight.ContainsKey(x) == false)
            {
                quadsInLevelThatBlockLight.Add(x, new Dictionary<int, List<QuadDefinition>>());
            }
            if (quadsInLevelThatBlockLight[x].ContainsKey(y) == false)
            {
                quadsInLevelThatBlockLight[x].Add(y, new List<QuadDefinition>());
            }
            quadsInLevelThatBlockLight[x][y].Add(quad);
        }


        private ICollection<QuadDefinition> GetQuadsToSeach(int qx, int qy, int d)
        {
            List<QuadDefinition> quads = new List<QuadDefinition>();

            for(int x = (qx - d); x <= (qx + d); x++)
            {
                for(int y = (qy - d); y <= (qy + d); y++)
                {

                    if(quadsInLevel.ContainsKey(x))
                    {
                        if (quadsInLevel[x].ContainsKey(y))
                        {
                            quads.AddRange(quadsInLevel[x][y]);
                        }
                    }

                }
            }

            return quads;
        }


        private ICollection<QuadDefinition> GetQuadsToSeachForLighting(int qx, int qy, int d)
        {
            List<QuadDefinition> quads = new List<QuadDefinition>();

            for (int x = (qx - d); x <= (qx + d); x++)
            {
                for (int y = (qy - d); y <= (qy + d); y++)
                {

                    if (quadsInLevelThatBlockLight.ContainsKey(x))
                    {
                        if (quadsInLevelThatBlockLight[x].ContainsKey(y))
                        {
                            quads.AddRange(quadsInLevelThatBlockLight[x][y]);
                        }
                    }

                }
            }

            return quads;
        }


        private void AssignLighting(LevelDefinition level)
		{
            foreach (LightDefinition light in level.Lights)
            {
                int c = 0;

                int x = (int)light.Position.X;
                int y = (int)light.Position.Y;
                int d = (int)(light.Brightness * 2);
                ICollection<QuadDefinition> quadsToSearch = GetQuadsToSeachForLighting(x, y, d);

                foreach (QuadDefinition quad in quadsToSearch)
                {
                    if (IsQuadWithinDistance(quad, light.Position, light.Brightness))
                    {
                        bool centroid = IsRayBlocked(quadsToSearch, quad, light.Position, quad.Centroid);
                        //bool p1 = IsRayBlocked(quadsToSearch, quad, light.Position, quad.Points[0]);
                        //bool p2 = IsRayBlocked(quadsToSearch, quad, light.Position, quad.Points[1]);
                        //bool p3 = IsRayBlocked(quadsToSearch, quad, light.Position, quad.Points[2]);
                        //bool p4 = IsRayBlocked(quadsToSearch, quad, light.Position, quad.Points[3]);
                        bool p1 = IsRayBlocked(quadsToSearch, quad, light.Position, GetPointCloserToCentroid(quad.Points[0], quad.Centroid, 1));
                        bool p2 = IsRayBlocked(quadsToSearch, quad, light.Position, GetPointCloserToCentroid(quad.Points[1], quad.Centroid, 1));
                        bool p3 = IsRayBlocked(quadsToSearch, quad, light.Position, GetPointCloserToCentroid(quad.Points[2], quad.Centroid, 1));
                        bool p4 = IsRayBlocked(quadsToSearch, quad, light.Position, GetPointCloserToCentroid(quad.Points[3], quad.Centroid, 1));

                        bool blocked = (centroid && p1 && p2 && p3 && p4);
                        //bool blocked = centroid;

                        if (blocked == false)
                        {
                            quad.LightKeys.Add(light.Key);
                            c++;
                        }
                    }
                }
                
                //System.Console.Out.WriteLine("NUM QUADS: " + c);
            }
		}


        private Vector3 GetPointCloserToCentroid(Vector3 point, Vector3 centroid, float step)
        {
            return Vector3.Lerp(point, centroid, step);
        }


        private bool IsQuadWithinDistance(QuadDefinition quad, Vector3 point, float distance)
        {
            float d = Vector3.Distance(quad.Centroid, point);

            return (d <= distance);
        }


        private bool IsRayBlocked(ICollection<QuadDefinition> quadsToSearch, QuadDefinition targetQuad, Vector3 light, Vector3 targetPoint)
        {
            Vector3 direction = Vector3.Normalize(targetPoint - light);
            float quadDistance = Vector3.Distance(targetPoint, light);

            foreach (QuadDefinition testingQuad in quadsToSearch)
            {
                //float testingQuadDistance = Vector3.Distance(testingQuad.Centroid, light);
                //for(int i = 0; i < 4; i++)
                //{
                //    float d = Vector3.Distance(testingQuad.Points[i], light);

                //    if (d > testingQuadDistance) testingQuadDistance = d;
                //}


                //if ((targetQuad != testingQuad) && (quadDistance >= testingQuadDistance))
                if (targetQuad != testingQuad)
                {
                    float factor;
                    Vector3 diff = Vector3.Zero;
                    Vector3 intersection = Vector3.Zero;
                    Vector2 p = Vector2.Zero;
                    Vector2 p1 = Vector2.Zero, p2 = Vector2.Zero, p3 = Vector2.Zero, p4 = Vector2.Zero;

                    bool intersected = false;

                    if (testingQuad.IsXAxisEqual())
                    {
                        factor = MathF.Abs((testingQuad.Points[0].X - light.X) / direction.X);
                        diff = direction * factor;
                        intersection = light + diff;

                        if (intersection.X == testingQuad.Points[0].X)
                        {
                            p = new Vector2(intersection.Y, intersection.Z);

                            p1 = new Vector2(testingQuad.Points[0].Y, testingQuad.Points[0].Z);
                            p2 = new Vector2(testingQuad.Points[1].Y, testingQuad.Points[1].Z);
                            p3 = new Vector2(testingQuad.Points[2].Y, testingQuad.Points[2].Z);
                            p4 = new Vector2(testingQuad.Points[3].Y, testingQuad.Points[3].Z);

                            intersected = true;
                        }
                    }
                    else if (testingQuad.IsYAxisEqual())
                    {
                        factor = MathF.Abs((testingQuad.Points[0].Y - light.Y) / direction.Y);
                        diff = direction * factor;
                        intersection = light + diff;

                        if (intersection.Y == testingQuad.Points[0].Y)
                        {
                            p = new Vector2(intersection.X, intersection.Z);

                            p1 = new Vector2(testingQuad.Points[0].X, testingQuad.Points[0].Z);
                            p2 = new Vector2(testingQuad.Points[1].X, testingQuad.Points[1].Z);
                            p3 = new Vector2(testingQuad.Points[2].X, testingQuad.Points[2].Z);
                            p4 = new Vector2(testingQuad.Points[3].X, testingQuad.Points[3].Z);

                            intersected = true;
                        }
                    }
                    else if (testingQuad.IsZAxisEqual())
                    {
                        factor = MathF.Abs((testingQuad.Points[0].Z - light.Z) / direction.Z);
                        diff = direction * factor;
                        intersection = light + diff;

                        if (intersection.Z == testingQuad.Points[0].Z)
                        {
                            p = new Vector2(intersection.X, intersection.Y);

                            p1 = new Vector2(testingQuad.Points[0].X, testingQuad.Points[0].Y);
                            p2 = new Vector2(testingQuad.Points[1].X, testingQuad.Points[1].Y);
                            p3 = new Vector2(testingQuad.Points[2].X, testingQuad.Points[2].Y);
                            p4 = new Vector2(testingQuad.Points[3].X, testingQuad.Points[3].Y);

                            intersected = true;
                        }
                    }

                    if (intersected && IsPointInRectangle(p, p1, p2, p3, p4))
                    {
                        float testingQuadDistance = Vector3.Distance(intersection, light);

                        if(testingQuadDistance < quadDistance) return true;
                    }
                }
                
            }

            return false;
        }


        private bool IsPointInRectangle(Vector2 p, Vector2 p1, Vector2 p2, Vector2 p3, Vector2 p4)
        {
            bool inX = false;
            bool inY = false;

            //X
            if (p1.X != p2.X)
            {
                if (p1.X < p2.X)
                {
                    inX = ((p.X >= p1.X) && (p.X <= p2.X));
                }
                else
                {
                    inX = ((p.X >= p2.X) && (p.X <= p1.X));
                }
            }
            else if (p1.X != p3.X)
            {
                if (p1.X < p3.X)
                {
                    inX = ((p.X >= p1.X) && (p.X <= p3.X));
                }
                else
                {
                    inX = ((p.X >= p3.X) && (p.X <= p1.X));
                }
            }
            else if (p1.X != p4.X)
            {
                if (p1.X < p4.X)
                {
                    inX = ((p.X >= p1.X) && (p.X <= p4.X));
                }
                else
                {
                    inX = ((p.X >= p4.X) && (p.X <= p1.X));
                }
            }

            //Y
            if (p1.Y != p2.Y)
            {
                if (p1.Y < p2.Y)
                {
                    inY = ((p.Y >= p1.Y) && (p.Y <= p2.Y));
                }
                else
                {
                    inY = ((p.Y >= p2.Y) && (p.Y <= p1.Y));
                }
            }
            else if (p1.Y != p3.Y)
            {
                if (p1.Y < p3.Y)
                {
                    inY = ((p.Y >= p1.Y) && (p.Y <= p3.Y));
                }
                else
                {
                    inY = ((p.Y >= p3.Y) && (p.Y <= p1.Y));
                }
            }
            else if (p1.Y != p4.Y)
            {
                if (p1.Y < p4.Y)
                {
                    inY = ((p.Y >= p1.Y) && (p.Y <= p4.Y));
                }
                else
                {
                    inY = ((p.Y >= p4.Y) && (p.Y <= p1.Y));
                }
            }

            return (inX && inY);
        }


    }
}

