﻿using Avalonia.Media.Imaging;
using RawResources;
using RawResources.Models;
using RawResources.Models.Levels;
using RawResources.Models.Resources;
using RawResources.Repositories;

namespace GameEditor.Models
{
	public class Session
	{
		public string WorkingDirectory { get; set; }
		public IRawContentManager? RawContentManager { get; set; }
		public IDictionary<int, bool> LevelsNeedCompiling { get; private set; }
		private IDictionary<int, IBitmap> textureBitmaps;
		private IBitmap? lightBitmap;
        private IBitmap? startPointBitmap;
        private IBitmap? teleportationPointBitmap;
		private IBitmap? pickupBitmap;
		private IBitmap? enemyBitmap;
		private IBitmap? victoryBitmap;
        private IBitmap? textBitmap;

        public Session()
		{
			this.textureBitmaps = new Dictionary<int, IBitmap>();
			this.WorkingDirectory = string.Empty;
			this.LevelsNeedCompiling = new Dictionary<int, bool>();
		}


		public bool Load(string workingDirectory)
		{
			string temp = workingDirectory.Replace("\\", "/");

			this.WorkingDirectory = temp;

			this.RawContentManager = new DefaultRawContentManager(this.WorkingDirectory);

			DetectLevelNeedsCompilationStatus();

            FindNewResources();

			return true;
		}


		public bool Save()
		{
			if (this.RawContentManager == null) return false;

			try
			{
				this.RawContentManager.Save();
			}
			catch(Exception)
			{
				return false;
			}

			return true;
		}


		public void SetLevelNeedsCompilationStatus(int levelKey, bool state)
		{
			if(state)
			{
				if (this.LevelsNeedCompiling.ContainsKey(levelKey) == false)
				{
					this.LevelsNeedCompiling.Add(levelKey, true);
				}
			}
			else
			{
                if (this.LevelsNeedCompiling.ContainsKey(levelKey) == true)
				{
					this.LevelsNeedCompiling.Remove(levelKey);
				}
            }
		}


		private void DetectLevelNeedsCompilationStatus()
		{
			if (this.RawContentManager == null) return;

            this.LevelsNeedCompiling.Clear();
            foreach (LevelDefinition levelDefinition in this.RawContentManager.LevelRepository.GetAllContent())
            {
                if (levelDefinition.HasQuadsInGeometry() == false)
                {
					this.LevelsNeedCompiling.Add(levelDefinition.Key, true);
                }
            }
        }


		private void FindNewResources()
		{
			MakePlayer();
			FindNewTextures();
			FindNewSongs();
			FindNewSounds();
			FindNewFonts();
		}


		private void MakePlayer()
		{
			if (this.RawContentManager == null) return;

			if(this.RawContentManager.PlayerRepository.GetKeys().Count() == 0)
			{
				PlayerDefinition definition = new PlayerDefinition()
				{
					Key = 0
				};

				this.RawContentManager.PlayerRepository.AddContent(definition);
			}
		}


		private void FindNewFonts()
		{
            if (this.RawContentManager == null) return;

            IRawContentRepository<FontDefinition> repo = this.RawContentManager.FontRepository;
            string path = this.WorkingDirectory + "/";

            IEnumerable<string> files;
            try
            {
                files = Directory.EnumerateFiles(path + "fonts", "*", SearchOption.AllDirectories);
            }
            catch (Exception)
            {
                return;
            }

            foreach (string file in files)
            {
                FileInfo fileInfo = new FileInfo(file);

                if (fileInfo.Attributes != FileAttributes.Hidden)
                {
                    string name = fileInfo.FullName.Replace("\\", "/").Replace(path, "");
                    bool found = false;

                    foreach (FontDefinition resource in repo.GetAllContent())
                    {
                        if (resource.Filename.Equals(name))
                        {
                            found = true;
                        }
                    }

                    if (found == false)
                    {
                        FontDefinition newResource = new FontDefinition()
                        {
                            Key = -1,
                            Filename = name,
                        };

						newResource.QuickName = newResource.ToString();

                        repo.AddContent(newResource);
                    }
                }
            }
        }


		private void FindNewTextures()
		{
			if (this.RawContentManager == null) return;

			IRawContentRepository<GraphicDefinition> repo = this.RawContentManager.GraphicRepository;
			string path = this.WorkingDirectory + "/";

			IEnumerable<string> files;
			try
			{
				files = Directory.EnumerateFiles(path + "graphics", "*", SearchOption.AllDirectories);
			}
			catch (Exception)
			{
				return;
			}

			foreach (string file in files)
			{
				FileInfo fileInfo = new FileInfo(file);

				if (fileInfo.Attributes != FileAttributes.Hidden)
				{
					string name = fileInfo.FullName.Replace("\\", "/").Replace(path, "");
					bool found = false;

					foreach (GraphicDefinition resource in repo.GetAllContent())
					{
						if (resource.Filename.Equals(name))
						{
							found = true;
						}
					}

					if (found == false)
					{
						var image = SixLabors.ImageSharp.Image.Load(fileInfo.FullName);

						GraphicDefinition newResource = new GraphicDefinition()
						{
							Key = -1,
							Filename = name,
							Size = image.Width
						};

						repo.AddContent(newResource);
					}
				}
			}
		}


		private void FindNewSongs()
		{
			if (this.RawContentManager == null) return;

			IRawContentRepository<SoundDefinition> repo = this.RawContentManager.SoundRepository;
			string path = this.WorkingDirectory + "/";

			IEnumerable<string> files;
			try
			{
				files = Directory.EnumerateFiles(path + "songs", "*", SearchOption.AllDirectories);
			}
			catch (Exception)
			{
				return;
			}

			foreach (string file in files)
			{
				FileInfo fileInfo = new FileInfo(file);

				if (fileInfo.Attributes != FileAttributes.Hidden)
				{
					string name = fileInfo.FullName.Replace("\\", "/").Replace(path, "");
					bool found = false;

					foreach (SoundDefinition resource in repo.GetAllContent())
					{
						if (resource.Filename.Equals(name))
						{
							found = true;
						}
					}

					if (found == false)
					{
						SoundDefinition newResource = new SoundDefinition()
						{
							Key = -1,
							Filename = name,
							IsLooped = true
						};

						repo.AddContent(newResource);
					}
				}
			}
		}


		private void FindNewSounds()
		{
			if (this.RawContentManager == null) return;

			IRawContentRepository<SoundDefinition> repo = this.RawContentManager.SoundRepository;
			string path = this.WorkingDirectory + "/";

			IEnumerable<string> files;
			try
			{
				files = Directory.EnumerateFiles(path + "sounds", "*", SearchOption.AllDirectories);
			}
			catch (Exception)
			{
				return;
			}

			foreach (string file in files)
			{
				FileInfo fileInfo = new FileInfo(file);

				if (fileInfo.Attributes != FileAttributes.Hidden)
				{
					string name = fileInfo.FullName.Replace("\\", "/").Replace(path, "");
					bool found = false;

					foreach (SoundDefinition resource in repo.GetAllContent())
					{
						if (resource.Filename.Equals(name))
						{
							found = true;
						}
					}

					if (found == false)
					{
						SoundDefinition newResource = new SoundDefinition()
						{
							Key = -1,
							Filename = name,
							IsLooped = false
						};

						repo.AddContent(newResource);
					}
				}
			}
		}


		public IBitmap? GetBitmapForTexture(int textureKey)
		{
			if (this.RawContentManager == null) return null;

			if (this.textureBitmaps.ContainsKey(textureKey)) return this.textureBitmaps[textureKey];

			GraphicDefinition? textureDefinition = this.RawContentManager.GraphicRepository.GetContent(textureKey);
			if (textureDefinition == null) return null;

			string path = this.WorkingDirectory + "/" + textureDefinition.Filename;
			IBitmap bitmap = new Bitmap(path);

			this.textureBitmaps.Add(textureKey, bitmap);

			return bitmap;
        }


		public IBitmap GetBitmapForLight()
		{
			if (this.lightBitmap != null) return this.lightBitmap;

			string path = "Icons/lightbulb.png";
            this.lightBitmap = new Bitmap(path);

            return this.lightBitmap;
		}


        public IBitmap GetBitmapForTextPoint()
        {
            if (this.textBitmap != null) return this.textBitmap;

            string path = "Icons/comment.png";
            this.textBitmap = new Bitmap(path);

            return this.textBitmap;
        }


        public IBitmap GetBitmapForStartPoint()
        {
            if (this.startPointBitmap != null) return this.startPointBitmap;

            string path = "Icons/user.png";
            this.startPointBitmap = new Bitmap(path);

            return this.startPointBitmap;
        }


        public IBitmap GetBitmapForVictoryPoint()
        {
            if (this.victoryBitmap != null) return this.victoryBitmap;

            string path = "Icons/star.png";
            this.victoryBitmap = new Bitmap(path);

            return this.victoryBitmap;
        }


        public IBitmap GetBitmapForTeleportationPoint()
        {
            if (this.teleportationPointBitmap != null) return this.teleportationPointBitmap;

            string path = "Icons/lightning.png";
            this.teleportationPointBitmap = new Bitmap(path);

            return this.teleportationPointBitmap;
        }


		public IBitmap GetBitmapForPickup()
		{
			if (this.pickupBitmap != null) return this.pickupBitmap;

			string path = "Icons/box_open.png";
			this.pickupBitmap = new Bitmap(path);

			return this.pickupBitmap;

        }


        public IBitmap GetBitmapForEnemy()
        {
            if (this.enemyBitmap != null) return this.enemyBitmap;

            string path = "Icons/user_death.png";
            this.enemyBitmap = new Bitmap(path);

            return this.enemyBitmap;
        }

    }
}

