﻿using System.Numerics;
using Avalonia.Controls;
using Avalonia.Markup.Xaml;
using AvaloniaColorPicker;
using GameEditor.Helpers;
using GameEditor.Models;
using GameEditor.Windows;
using RawResources.Models.Levels;

namespace GameEditor.Panels.LevelEditor
{
    public class LightPropertiesPanel : UserControl
    {
        private LevelEditorWindow? levelEditorWindow;
        private Session? session;
        

        public LightPropertiesPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);
        }


        public Vector3 Color { get { return ColorHelper.GetVector(this.FindControl<ColorButton>("ColorChoice").Color); } }
        public float Brightness { get { return (float)this.FindControl<NumericUpDown>("BrightnessTextBox").Value; } }


        public void SetArguments(LevelEditorWindow levelEditorWindow, Session session)
        {
            this.levelEditorWindow = levelEditorWindow;
            this.session = session;

            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            ColorButton colorChoice = this.FindControl<ColorButton>("ColorChoice");
            colorChoice.Color = new Avalonia.Media.Color(255, 255, 255, 255);
            colorChoice.PropertyChanged += ColorChoice_PropertyChanged;

            NumericUpDown brightnessTextBox = this.FindControl<NumericUpDown>("BrightnessTextBox");
            brightnessTextBox.Value = 100;
            brightnessTextBox.ValueChanged += BrightnessTextBox_ValueChanged;

            UpdateSelected();
        }


        private void BrightnessTextBox_ValueChanged(object? sender, NumericUpDownValueChangedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedLight != null)
            {
                this.levelEditorWindow.SelectedLight.Brightness = this.Brightness;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<LightDefinition> lights = GetMultipleSelected();

                foreach (LightDefinition light in lights)
                {
                    light.Brightness = this.Brightness;
                }
            }
        }


        private void ColorChoice_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (e.Property == ColorButton.ColorProperty)
            {
                if (this.levelEditorWindow.SelectedLight != null)
                {
                    this.levelEditorWindow.SelectedLight.Color = this.Color;
                }
                else if (this.levelEditorWindow.SelectedObjects != null)
                {
                    IEnumerable<LightDefinition> lights = GetMultipleSelected();

                    foreach(LightDefinition light in lights)
                    {
                        light.Color = this.Color;
                    }
                }
            }
        }


        public void UpdateSelected()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;
            if (this.levelEditorWindow == null) return;

            TextBlock selectionText = this.FindControl<TextBlock>("SelectionText");
            ColorButton colorChoice = this.FindControl<ColorButton>("ColorChoice");
            NumericUpDown brightnessTextBox = this.FindControl<NumericUpDown>("BrightnessTextBox");

            if (this.levelEditorWindow.SelectedLight != null)
            {
                colorChoice.Color = ColorHelper.GetAvaloniaColor(this.levelEditorWindow.SelectedLight.Color);
                brightnessTextBox.Value = this.levelEditorWindow.SelectedLight.Brightness;

                selectionText.Text = "(currently selected)";
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                selectionText.Text = "(multiple selected)";
            }
            else
            {
                selectionText.Text = "(nothing selected)";
            }
        }


        private IEnumerable<LightDefinition> GetMultipleSelected()
        {
            if (this.levelEditorWindow == null) return new List<LightDefinition>();
            if (this.levelEditorWindow.SelectedObjects == null) return new List<LightDefinition>();

            IEnumerable<LightDefinition> lights = this.levelEditorWindow.SelectedObjects.Where(x => x.GetType() == typeof(LightDefinition)).Select(x => (LightDefinition)x);

            return lights;
        }



    }
}

