﻿using System.Xml.Linq;
using Avalonia;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameEditor.Models;
using GameEditor.Windows;
using RawResources.Models.Items;
using RawResources.Models.Levels;
using static System.Net.Mime.MediaTypeNames;

namespace GameEditor.Panels.LevelEditor
{
    public class NodePropertiesPanel : UserControl
    {
        private LevelEditorWindow? levelEditorWindow;
        private Session? session;
        

        public NodePropertiesPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.FindControl<Button>("ClearChoice").Click += ClearChoice_Click;
        }


        public NodeTypes NodeType
        {
            get
            {
                NodeTypes? nodeType = (NodeTypes?)this.FindControl<ComboBox>("TypeList").SelectedItem;

                if (nodeType != null) return nodeType.Value;
                else return NodeTypes.StartPoint;
            }
        }


        public void SetArguments(LevelEditorWindow levelEditorWindow, Session session)
        {
            this.levelEditorWindow = levelEditorWindow;
            this.session = session;

            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            ComboBox comboBox = this.FindControl<ComboBox>("TypeList");
            comboBox.Items = Enum.GetValues<NodeTypes>();
            comboBox.SelectedItem = NodeTypes.StartPoint;

            comboBox = this.FindControl<ComboBox>("ChoiceList");
            comboBox.Items = null;
            comboBox.SelectedItem = null;

            AddEvents();

            UpdateSelected();
        }


        private void RemoveEvents()
        {
            this.FindControl<ComboBox>("TypeList").SelectionChanged -= TypeList_SelectionChanged;
            this.FindControl<ComboBox>("ChoiceList").SelectionChanged -= ChoiceList_SelectionChanged;
            this.FindControl<TextBox>("Text").PropertyChanged -= Text_PropertyChanged;
        }


        private void AddEvents()
        {
            this.FindControl<ComboBox>("TypeList").SelectionChanged += TypeList_SelectionChanged;
            this.FindControl<ComboBox>("ChoiceList").SelectionChanged += ChoiceList_SelectionChanged;
            this.FindControl<TextBox>("Text").PropertyChanged += Text_PropertyChanged;
        }


        private void ClearChoice_Click(object? sender, RoutedEventArgs e)
        {
            ComboBox comboBox = this.FindControl<ComboBox>("ChoiceList");
            comboBox.SelectedItem = null;
        }


        private void Text_PropertyChanged(object? sender, AvaloniaPropertyChangedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            TextBox text = this.FindControl<TextBox>("Text");

            //Get Nodes
            List<NodeDefinition> nodes = new List<NodeDefinition>();

            if (this.levelEditorWindow.SelectedNode != null)
            {
                nodes.Add(this.levelEditorWindow.SelectedNode);
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                nodes.AddRange(GetMultipleSelected());
            }

            //Update Nodes
            foreach (NodeDefinition node in nodes)
            {
                node.Text = text.Text;
            }
        }


        private void TypeList_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            RemoveEvents();

            //Get Nodes
            List<NodeDefinition> nodes = new List<NodeDefinition>();

            if (this.levelEditorWindow.SelectedNode != null)
            {
                nodes.Add(this.levelEditorWindow.SelectedNode);
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                nodes.AddRange(GetMultipleSelected());
            }

            //Update Nodes
            foreach (NodeDefinition node in nodes)
            {
                node.NodeType = this.NodeType;
                node.ChoiceKey = -1;
            }

            //Load Choice List
            LoadChoiceList(this.NodeType);

            SetEnability(this.NodeType);

            AddEvents();

            this.levelEditorWindow.RefreshVisuals();
        }


        private void ChoiceList_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            ComboBox comboBox = this.FindControl<ComboBox>("ChoiceList");
            NodeTypes nodeType = this.NodeType;

            //Get Choice Key
            int key = -1;

            if(nodeType == NodeTypes.Pickup)
            {
                PickupDefinition? pickupDefinition = (PickupDefinition?)comboBox.SelectedItem;
                if (pickupDefinition != null) key = pickupDefinition.Key;
            }
            else if (nodeType == NodeTypes.Enemy)
            {
                EnemyDefinition? enemyDefinition = (EnemyDefinition?)comboBox.SelectedItem;
                if (enemyDefinition != null) key = enemyDefinition.Key;
            }

            //Get Nodes
            List<NodeDefinition> nodes = new List<NodeDefinition>();

            if (this.levelEditorWindow.SelectedNode != null)
            {
                nodes.Add(this.levelEditorWindow.SelectedNode);
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                nodes.AddRange(GetMultipleSelected());
            }

            //Update Nodes
            foreach (NodeDefinition node in nodes)
            {
                node.ChoiceKey = key;
            }

            this.levelEditorWindow.RefreshVisuals();
        }


        private void LoadChoiceList(NodeTypes nodeType)
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            ComboBox comboBox = this.FindControl<ComboBox>("ChoiceList");

            if (nodeType == NodeTypes.Pickup)
            {
                comboBox.Items = this.session.RawContentManager.PickupRepository.GetAllContent();
            }
            else if (nodeType == NodeTypes.Enemy)
            {
                comboBox.Items = this.session.RawContentManager.EnemyRepository.GetAllContent();
            }
            else
            {
                comboBox.Items = null;
            }
        }


        public void UpdateSelected()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;
            if (this.levelEditorWindow == null) return;

            RemoveEvents();

            TextBlock selectionText = this.FindControl<TextBlock>("SelectionText");
            ComboBox nodeTypeList = this.FindControl<ComboBox>("TypeList");
            ComboBox choiceList = this.FindControl<ComboBox>("ChoiceList");
            TextBox text = this.FindControl<TextBox>("Text");

            if (this.levelEditorWindow.SelectedNode != null)
            {
                NodeDefinition node = this.levelEditorWindow.SelectedNode;

                nodeTypeList.SelectedItem = node.NodeType;
                LoadChoiceList(node.NodeType);

                if(node.NodeType == NodeTypes.Pickup)
                {
                    PickupDefinition? pickupDefinition = this.session.RawContentManager.PickupRepository.GetContent(node.ChoiceKey);
                    choiceList.SelectedItem = pickupDefinition;

                    text.Text = "";
                }
                else if (node.NodeType == NodeTypes.Enemy)
                {
                    EnemyDefinition? enemyDefinition = this.session.RawContentManager.EnemyRepository.GetContent(node.ChoiceKey);
                    choiceList.SelectedItem = enemyDefinition;

                    text.Text = "";
                }
                else if (node.NodeType == NodeTypes.Text)
                {
                    choiceList.SelectedItem = null;

                    text.Text = node.Text;
                }

                SetEnability(node.NodeType);

                selectionText.Text = "(currently selected #" + this.levelEditorWindow.SelectedNode.Key + ")";
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                selectionText.Text = "(multiple selected)";
            }
            else
            {
                selectionText.Text = "(nothing selected)";
            }

            AddEvents();
        }


        private void SetEnability(NodeTypes nodeType)
        {
            if (this.levelEditorWindow == null) return;

            Button clearButton = this.FindControl<Button>("ClearChoice");
            ComboBox nodeTypeList = this.FindControl<ComboBox>("TypeList");
            ComboBox choiceList = this.FindControl<ComboBox>("ChoiceList");
            TextBox text = this.FindControl<TextBox>("Text");

            this.levelEditorWindow.Focus();

            nodeTypeList.IsEnabled = true;

            if (nodeType == NodeTypes.Pickup)
            {
                choiceList.IsEnabled = true;
                clearButton.IsEnabled = true;
                text.IsEnabled = false;
            }
            else if (nodeType == NodeTypes.Enemy)
            {
                choiceList.IsEnabled = true;
                clearButton.IsEnabled = true;
                text.IsEnabled = false;
            }
            else if (nodeType == NodeTypes.Text)
            {
                choiceList.IsEnabled = false;
                clearButton.IsEnabled = false;
                text.IsEnabled = true;
            }
            else
            {
                choiceList.IsEnabled = false;
                clearButton.IsEnabled = false;
                text.IsEnabled = false;
            }
        }


        private IEnumerable<NodeDefinition> GetMultipleSelected()
        {
            if (this.levelEditorWindow == null) return new List<NodeDefinition>();
            if (this.levelEditorWindow.SelectedObjects == null) return new List<NodeDefinition>();

            IEnumerable<NodeDefinition> nodes = this.levelEditorWindow.SelectedObjects.Where(x => x.GetType() == typeof(NodeDefinition)).Select(x => (NodeDefinition)x);

            return nodes;
        }
        



        
    }
}

